#' Dynamic material modeling from strain rate temperature table
#'
#' @description Dynamic material modeling based on strain rate-temperature table returned from
#' the function \code{\link[TPMplt:epsExtract]{epsExtract}}. Material constants as well as power
#' dissipation efficiency factors and rheological stability coefficients in current conditions
#' will be returned.
#' @param x A strain rate-temperature table, returned from \code{\link[TPMplt:epsExtract]{epsExtract}}.
#' @param consfuncPRT A boolean value to control result printing for constructive function. The
#' default value uses FALSE.
#' @param lgbase A numeric value to determine the logarithm base in calculation. The default
#' value is exp(1).
#' @param rnd An integer to control the digit accuracy. Default setting is 2, means accurating to 0.01.
#'
#' @return Serial material constants, constructive function, eta table and xi table through
#' dynamic material model developed by Gegel and Prasad.
#' @import VBTree stats
#' @export DMMprocess
#' @seealso \code{\link[VBTree:VBTree-package]{VBTree}}, \code{\link[TPMplt:epsExtract]{epsExtract}}
#'
#' @examples
#' require(VBTree)
#' dl2vbt(chrvec2dl(colnames(TPMdata)))
#' epstable <- epsExtract(TPMdata, 0.7, 2, 3)
#' DMM <- DMMprocess(epstable)
#' DMM
#' @keywords DMMprocess DMMresult epsExtract
DMMprocess <- function(x, consfuncPRT=FALSE, lgbase=exp(1), rnd=2){

  # check the input data
  if(class(x)!="SR-T.table"){
    stop("the input data should be SR-T Chart generated by the function epsinprt()", call. = FALSE)
  }

  # # test section
  # base <- exp(1)
  # x <- epstable
  # eigenPRT=TRUE

  eps <- x$epsilon
  x <- x$SRT.table

  # T1 to T in function
  logbase <- lgbase

  dims <- dim(x)
  SR <- as.numeric(rownames(x))
  Tem1 <- 1/as.numeric(colnames(x))
  lgSR <- log(SR, base = logbase)

  i <- 1
  temp_vec <- c()
  for (i in 1:dims[2]) {
    b <- as.vector(log(x[,i], base = logbase))
    a <- lgSR
    slope <- as.numeric(lm(b~I(a^1))$coefficients[2])
    temp_vec <- append(temp_vec, slope)
  }

  m.SRSensitivity <- temp_vec # vector; since m ~ f(T)
  n1.StressInd <- 1/mean(m.SRSensitivity)
  #eta.DissiEfficiFactor <- (2*m.SRSensitivity)/(1+m.SRSensitivity) # vector; eta ~ f(T)

  i <- 1
  temp_vec <- c()
  for (i in 1:dims[2]) {
    b <- as.vector(x[,i])
    a <- lgSR
    slope <- as.numeric(lm(b~I(a^1))$coefficients[2])
    temp_vec <- append(temp_vec, slope)
  }

  beta.StressInd <- mean(1/temp_vec)
  alpha.MateCons <- beta.StressInd/n1.StressInd # unit: MPa^(-1)

  lgsinhx <- log(sinh(alpha.MateCons*x), base = logbase) # make the log(sinh[alpha*sigma]) values

  i <- 1
  temp_vec <- c()
  for (i in 1:dims[1]) {
    b <- as.vector(lgsinhx[i,])
    a <- Tem1
    slope <- as.numeric(lm(b~I(a^1))$coefficients[2])
    temp_vec <- append(temp_vec, slope)
  }

  K.MetaCons <- temp_vec

  i <- 1
  temp_vec <- c()
  for (i in 1:dims[2]) {
    b <- lgSR
    a <- as.vector(lgsinhx[,i])
    slope <- as.numeric(lm(b~I(a^1))$coefficients[2])
    temp_vec <- append(temp_vec, slope)
  }

  Hem1.MetaCons <- temp_vec

  R.Cons <- 8.314
  Q.Temp <- R.Cons *mean(K.MetaCons) *mean(Hem1.MetaCons)

  # traverse dims[1]*dims[2] times using the built-in order in R
  i <- 1
  j <- 1
  Zmat <- matrix(NA, nrow = dims[1], ncol = dims[2])
  for (i in 1:dims[1]) {
    for (j in 1:dims[2]) {
      Tcond <- as.numeric(colnames(lgsinhx)[j])
      SRcond <- as.numeric(rownames(lgsinhx)[i])
      temp <- exp(Q.Temp/(R.Cons*Tcond))
      Zmat[i,j] <- SRcond*temp
    }
  }

  Z <- as.numeric(as.vector(Zmat))
  lnZ <- log(Z, base = logbase)
  a1 <- as.numeric(lgsinhx)

  n.PowerValue <- as.numeric(lm(lnZ~I(a1^1))$coefficients[2])

  a2 <- as.numeric(as.vector((sinh(alpha.MateCons*x))^n.PowerValue))
  A.MetaCons <- as.numeric(lm(Z~I(a2^1))$coefficients[2])

  Q.ActivEnerg <- Q.Temp*(-0.001) # unit: kJ/mol

  if(consfuncPRT){
    print(paste("the constitutive equation in strain ", round(eps, rnd), " is: sigma=", "(1/", round(alpha.MateCons, rnd), ")ln{(Z/", A.MetaCons, ")^(1/", round(n.PowerValue, rnd),
                ")+[(Z/", A.MetaCons, ")^(2/", round(n.PowerValue, rnd), ")", "+1]^(1/2)}", sep =""))
  }

  # package all the coefficients
  MaterialCoefficients <- list("m.StrainRateSensitivity"=m.SRSensitivity, "n1.StressIndex"=n1.StressInd, "beta.StressIndex"=beta.StressInd,
                               "alpha.MaterialConstant"=alpha.MateCons, "Q.ActivatingEnergy"=Q.ActivEnerg, "n.PowerValue"=n.PowerValue,
                               "A.MaterialConstant"= A.MetaCons, "epsilon.strain"=eps, "base"=logbase)

  i <- 1
  fitMat <- matrix(NA, nrow = 3, ncol = dims[2])
  for (i in 1:dims[2]) {
    b <- as.vector(log(x[,i], base = logbase))
    a <- lgSR
    coeffivec <- as.vector(lm(b~I(a^1)+I(a^2)+I(a^3))$coefficients)
    fitMat[,i] <- c(coeffivec[2], coeffivec[3], coeffivec[4])
  }

  temptable <- matrix(NA, nrow = dims[1], ncol = dims[2])

  i <- 1
  j <- 1
  for (i in 1:dims[1]) { # max i = 4
    for (j in 1:dims[2]) { # max j = 7
      bcd <- fitMat[,j] # d3 vector
      lgstr <- log(as.numeric(rownames(lgsinhx)[i]), base = logbase)
      temptable[i,j] <- bcd[1]+2*bcd[2]*lgstr+3*bcd[3]*(lgstr^2)
    }
  }

  etatable <- as.data.frame((2*temptable)/(1-temptable))
  rownames(etatable) <- rownames(lgsinhx)
  colnames(etatable) <- colnames(lgsinhx)

  i <- 1
  temp_vec <- c()
  for (i in 1:dims[2]) {
    b <- temptable[,i]/(temptable[,i] + 1)
    a <- lgSR
    temptable[,i] <- as.numeric(lm(b~I(a^1))$coefficients[2]) + temptable[,i]
  }

  xitable <- temptable
  rownames(xitable) <- rownames(lgsinhx)
  colnames(xitable) <- colnames(lgsinhx)

  # package two tables
  rslttable <- list("SRTtable"=x, "etatable"=etatable, "xitable"=xitable)

  result <- list("MaterialCoefficients"=MaterialCoefficients, "tablelist"=rslttable)
  class(result) <- "DMMresult"
  return(result)
}
