/******************************************************************************
 * @section DESCRIPTION
 *
 * Calculate snow accumulation and melt
 *
 * @section LICENSE
 *
 * The Variable Infiltration Capacity (VIC) macroscale hydrological model
 * Copyright (C) 2016 The Computational Hydrology Group, Department of Civil
 * and Environmental Engineering, University of Washington.
 *
 * The VIC model is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *****************************************************************************/

#include <vic_run.h>

/******************************************************************************
 * @brief    Calculate snow accumulation and melt using an energy balance
 *           approach for a two layer snow model
 *****************************************************************************/
int
snow_melt(double            Le,
          double            NetShortSnow,       // net SW at absorbed by snow
          double            Tcanopy,
          double            Tgrnd,
          double           *Z0,        // roughness
          double            aero_resist,       // aerodynamic resistance
          double           *aero_resist_used,        // stability-corrected aerodynamic resistance
          double            air_temp,       // air temperature
          double            coverage,      // snowpack cover fraction
          double            delta_t,       // time step in secs
          double            density,       // atmospheric density
          double            grnd_flux,       // ground heat flux
          double            LongSnowIn,       // incoming longwave radiation
          double            pressure,
          double            rainfall,
          double            snowfall,
          double            vp,
          double            vpd,
          double            wind,
          double            z2,
          double           *NetLongSnow,
          double           *OldTSurf,
          double           *melt,
          double           *save_Qnet,
          double           *save_advected_sensible,
          double           *save_advection,
          double           *save_deltaCC,
          double           *save_grnd_flux,
          double           *save_latent,
          double           *save_latent_sub,
          double           *save_refreeze_energy,
          double           *save_sensible,
          int               UNSTABLE_SNOW,
          int               iveg,
          int               band,
          snow_data_struct *snow)
{
    extern option_struct     options;
    extern parameters_struct param;

    double                   error;
    double                   DeltaPackCC; /* Change in cold content of the pack */
    double                   DeltaPackSwq; /* Change in snow water equivalent of the
                                              pack (m) */
    double                   Ice; /* Ice content of snow pack (m)*/
    double                   InitialSwq; /* Initial snow water equivalent (m) */
    double                   MassBalanceError; /* Mass balance error (m) */
    double                   MaxLiquidWater; /* Maximum liquid water content of pack (m) */
    double                   PackCC; /* Cold content of snow pack (J) */
    double                   PackSwq; /* Snow pack snow water equivalent (m) */
    double                   Qnet; /* Net energy exchange at the surface (W/m2) */
    double                   RefreezeEnergy; /* refreeze/melt energy in surface layer (W/m2) */
    double                   PackRefreezeEnergy; /* refreeze/melt energy in pack layer (W/m2) */
    double                   RefrozenWater; /* Amount of refrozen water (m) */
    double                   SnowFallCC; /* Cold content of new snowfall (J) */
    double                   SnowMelt; /* Amount of snow melt during time interval
                                          (m water equivalent) */
    double                   SurfaceCC; /* Cold content of snow pack (J) */
    double                   SurfaceSwq; /* Surface layer snow water equivalent (m) */
    double                   SnowFall;
    double                   RainFall;
    double                   advection;
    double                   deltaCC;
    double                   latent_heat;
    double                   latent_heat_sub;
    double                   sensible_heat;
    double                   advected_sensible_heat;
    double                   melt_energy = 0.;

    SnowFall = snowfall / MM_PER_M; /* convet to m */
    RainFall = rainfall / MM_PER_M; /* convet to m */

    InitialSwq = snow->swq;
    (*OldTSurf) = snow->surf_temp;

    /* Initialize snowpack variables */

    Ice = snow->swq - snow->pack_water - snow->surf_water;

    /* Reconstruct snow pack */
    if (Ice > param.SNOW_MAX_SURFACE_SWE) {
        SurfaceSwq = param.SNOW_MAX_SURFACE_SWE;
    }
    else {
        SurfaceSwq = Ice;
    }
    PackSwq = Ice - SurfaceSwq;

    /* Calculate cold contents */
    SurfaceCC = CONST_VCPICE_WQ * SurfaceSwq * snow->surf_temp;
    PackCC = CONST_VCPICE_WQ * PackSwq * snow->pack_temp;
    if (air_temp > 0.0) {
        SnowFallCC = 0.0;
    }
    else {
        SnowFallCC = CONST_VCPICE_WQ * SnowFall * air_temp;
    }

    /* Distribute fresh snowfall */
    if (SnowFall > (param.SNOW_MAX_SURFACE_SWE - SurfaceSwq) &&
        (param.SNOW_MAX_SURFACE_SWE - SurfaceSwq) > DBL_EPSILON) {
        DeltaPackSwq = SurfaceSwq + SnowFall - param.SNOW_MAX_SURFACE_SWE;
        if (DeltaPackSwq > SurfaceSwq) {
            DeltaPackCC = SurfaceCC +
                          (SnowFall -
                           param.SNOW_MAX_SURFACE_SWE) / SnowFall * SnowFallCC;
        }
        else {
            DeltaPackCC = DeltaPackSwq / SurfaceSwq * SurfaceCC;
        }
        SurfaceSwq = param.SNOW_MAX_SURFACE_SWE;
        SurfaceCC += SnowFallCC - DeltaPackCC;
        PackSwq += DeltaPackSwq;
        PackCC += DeltaPackCC;
    }
    else {
        SurfaceSwq += SnowFall;
        SurfaceCC += SnowFallCC;
    }
    if (SurfaceSwq > 0.0) {
        snow->surf_temp = SurfaceCC / (CONST_VCPICE_WQ * SurfaceSwq);
    }
    else {
        snow->surf_temp = 0.0;
    }
    if (PackSwq > 0.0) {
        snow->pack_temp = PackCC / (CONST_VCPICE_WQ * PackSwq);
    }
    else {
        snow->pack_temp = 0.0;
    }

    /* Adjust ice and snow->surf_water */
    Ice += SnowFall;
    snow->surf_water += RainFall;

    /* Calculate the surface energy balance for snow_temp = 0.0 */

    Qnet = CalcSnowPackEnergyBalance((double) 0.0, delta_t, aero_resist,
                                     aero_resist_used, z2, Z0,
                                     density, vp, LongSnowIn, Le, pressure,
                                     RainFall, NetShortSnow, vpd,
                                     wind, (*OldTSurf), coverage,
                                     snow->depth, snow->density,
                                     snow->surf_water, SurfaceSwq,
                                     Tcanopy, Tgrnd,
                                     &advection, &advected_sensible_heat,
                                     &deltaCC,
                                     &grnd_flux, &latent_heat,
                                     &latent_heat_sub, NetLongSnow,
                                     &RefreezeEnergy, &sensible_heat,
                                     &snow->vapor_flux, &snow->blowing_flux,
                                     &snow->surface_flux);

    /* Check that snow swq exceeds minimum value for model stability */
    if (!UNSTABLE_SNOW) {
        /* If Qnet == 0.0, then set the surface temperature to 0.0 */
        if (Qnet == 0.0) {
            snow->surf_temp = 0.0;
            if (RefreezeEnergy >= 0.0) {
                RefrozenWater = RefreezeEnergy /
                                (CONST_LATICE * CONST_RHOFW) * delta_t;
                if (RefrozenWater > snow->surf_water) {
                    RefrozenWater = snow->surf_water;
                    RefreezeEnergy = RefrozenWater * CONST_LATICE *
                                     CONST_RHOFW / (delta_t);
                }
                melt_energy += RefreezeEnergy;
                SurfaceSwq += RefrozenWater;
                Ice += RefrozenWater;
                snow->surf_water -= RefrozenWater;
                if (snow->surf_water < 0.0) {
                    snow->surf_water = 0.0;
                }
                SnowMelt = 0.0;
            }
            else {
                /* Calculate snow melt */
                SnowMelt = fabs(RefreezeEnergy) /
                           (CONST_LATICE * CONST_RHOFW) * delta_t;
                melt_energy += RefreezeEnergy;
            }

            /* Adjust snow->surf_water for vapor_flux */
            if (snow->surf_water < -(snow->vapor_flux)) {
                // if vapor_flux exceeds surf_water, we not only need to
                // re-scale vapor_flux, we need to re-scale surface_flux and blowing_flux
                snow->blowing_flux *= -(snow->surf_water / snow->vapor_flux);
                snow->vapor_flux = -(snow->surf_water);
                snow->surface_flux = -(snow->surf_water) - snow->blowing_flux;
                snow->surf_water = 0.0;
            }
            else {
                snow->surf_water += snow->vapor_flux;
            }

            /* If SnowMelt < Ice, there was incomplete melting of the pack */

            if (SnowMelt < Ice) {
                if (SnowMelt <= PackSwq) {
                    snow->surf_water += SnowMelt;
                    PackSwq -= SnowMelt;
                    Ice -= SnowMelt;
                }
                else {
                    snow->surf_water += SnowMelt + snow->pack_water;
                    snow->pack_water = 0.0;
                    PackSwq = 0.0;
                    Ice -= SnowMelt;
                    SurfaceSwq = Ice;
                }
            }
            /* Else, SnowMelt > Ice and there was complete melting of the pack */
            else {
                SnowMelt = Ice;
                snow->surf_water += Ice;
                SurfaceSwq = 0.0;
                snow->surf_temp = 0.0;
                PackSwq = 0.0;
                snow->pack_temp = 0.0;
                Ice = 0.0;
                /* readjust melt energy to account for melt only of available snow */
                melt_energy -= RefreezeEnergy;
                RefreezeEnergy = RefreezeEnergy / fabs(RefreezeEnergy) *
                                 SnowMelt * CONST_LATICE * CONST_RHOFW /
                                 (delta_t);
                melt_energy += RefreezeEnergy;
            }
        }
        /* Else, SnowPackEnergyBalance(T=0.0) <= 0.0 */
        else {
            /* Calculate surface layer temperature using "Brent method" */
            if (SurfaceSwq > param.SNOW_MIN_SWQ_EB_THRES) {
                snow->surf_temp = root_brent(
                    (double) (snow->surf_temp - param.SNOW_DT),
                    (double) (snow->surf_temp + param.SNOW_DT),
                    SnowPackEnergyBalance,
                    delta_t, aero_resist, aero_resist_used, z2, Z0,
                    density, vp, LongSnowIn, Le, pressure,
                    RainFall, NetShortSnow, vpd,
                    wind, (*OldTSurf), coverage,
                    snow->depth, snow->density,
                    snow->surf_water, SurfaceSwq,
                    Tcanopy, Tgrnd,
                    &advection, &advected_sensible_heat,
                    &deltaCC,
                    &grnd_flux, &latent_heat,
                    &latent_heat_sub, NetLongSnow,
                    &RefreezeEnergy, &sensible_heat,
                    &snow->vapor_flux, &snow->blowing_flux,
                    &snow->surface_flux);

                if (snow->surf_temp <= -998) {
                    if (options.TFALLBACK) {
                        snow->surf_temp = *OldTSurf;
                        snow->surf_temp_fbflag = 1;
                        snow->surf_temp_fbcount++;
                    }
                    else {
                        error = ErrorSnowPackEnergyBalance(snow->surf_temp,
                                                           iveg, band,
                                                           delta_t, aero_resist,
                                                           aero_resist_used,
                                                           z2, Z0, density, vp,
                                                           LongSnowIn, Le,
                                                           pressure,
                                                           RainFall,
                                                           NetShortSnow, vpd,
                                                           wind, (*OldTSurf),
                                                           coverage,
                                                           snow->depth,
                                                           snow->density,
                                                           snow->surf_water,
                                                           SurfaceSwq,
                                                           Tcanopy, Tgrnd,
                                                           &advection,
                                                           &advected_sensible_heat,
                                                           &deltaCC,
                                                           &grnd_flux, &latent_heat,
                                                           &latent_heat_sub,
                                                           NetLongSnow, &RefreezeEnergy,
                                                           &sensible_heat, &snow->vapor_flux,
                                                           &snow->blowing_flux,
                                                           &snow->surface_flux);
                        return(error);
                    }
                }
            }
            else {
                /* Thin snowpack must be solved in conjunction with ground surface energy balance */
                snow->surf_temp = 999;
            }
            if (snow->surf_temp > -998 && snow->surf_temp < 999) {
                Qnet = CalcSnowPackEnergyBalance(snow->surf_temp,
                                                 delta_t, aero_resist,
                                                 aero_resist_used, z2, Z0,
                                                 density, vp, LongSnowIn, Le,
                                                 pressure,
                                                 RainFall, NetShortSnow, vpd,
                                                 wind, (*OldTSurf), coverage,
                                                 snow->depth, snow->density,
                                                 snow->surf_water, SurfaceSwq,
                                                 Tcanopy, Tgrnd,
                                                 &advection,
                                                 &advected_sensible_heat,
                                                 &deltaCC,
                                                 &grnd_flux, &latent_heat,
                                                 &latent_heat_sub,
                                                 NetLongSnow, &RefreezeEnergy,
                                                 &sensible_heat,
                                                 &snow->vapor_flux,
                                                 &snow->blowing_flux,
                                                 &snow->surface_flux);

                /* since we iterated, the surface layer is below freezing and no snowmelt */

                SnowMelt = 0.0;

                /* Since updated snow_temp < 0.0, all of the liquid water in the surface
                   layer has been frozen */

                SurfaceSwq += snow->surf_water;
                Ice += snow->surf_water;
                snow->surf_water = 0.0;
                melt_energy += snow->surf_water * CONST_LATICE * CONST_RHOFW /
                               (delta_t);

                /* Adjust SurfaceSwq for vapor flux */
                if (SurfaceSwq < -(snow->vapor_flux)) {
                    // if vapor_flux exceeds SurfaceSwq, we not only need to
                    // re-scale vapor_flux, we need to re-scale surface_flux and blowing_flux
                    snow->blowing_flux *= -(SurfaceSwq / snow->vapor_flux);
                    snow->vapor_flux = -SurfaceSwq;
                    snow->surface_flux = -SurfaceSwq - snow->blowing_flux;
                    SurfaceSwq = 0.0;
                    Ice = PackSwq;
                }
                else {
                    SurfaceSwq += snow->vapor_flux;
                    Ice += snow->vapor_flux;
                }
            }
        }
    }
    else {
        /* Snow solution is unstable as independent layer */
        snow->surf_temp = 999;
    }

    /* Done with iteration etc, now Update the liquid water content of the
       surface layer */

    MaxLiquidWater = param.SNOW_LIQUID_WATER_CAPACITY * SurfaceSwq;
    if (snow->surf_water > MaxLiquidWater) {
        melt[0] = snow->surf_water - MaxLiquidWater;
        snow->surf_water = MaxLiquidWater;
    }
    else {
        melt[0] = 0.0;
    }

    /* Refreeze liquid water in the pack.
       variable 'RefreezeEnergy' is the heat released to the snow pack
       if all liquid water were refrozen.
       if RefreezeEnergy < PackCC then all water IS refrozen
       PackCC always <=0.0

       WORK IN PROGRESS: This energy is NOT added to MeltEnergy, since this does
       not involve energy transported to the pixel.  Instead heat from the snow
       pack is used to refreeze water */

    snow->pack_water += melt[0]; /* add surface layer outflow to pack
                                        liquid water*/
    PackRefreezeEnergy = snow->pack_water * CONST_LATICE * CONST_RHOFW;

    /* calculate energy released to freeze*/

    if (PackCC < -PackRefreezeEnergy) { /* cold content not fully depleted*/
        PackSwq += snow->pack_water;  /* refreeze all water and update*/
        Ice += snow->pack_water;
        snow->pack_water = 0.0;
        if (PackSwq > 0.0) {
            PackCC = PackSwq * CONST_VCPICE_WQ * snow->pack_temp +
                     PackRefreezeEnergy;
            snow->pack_temp = PackCC / (CONST_VCPICE_WQ * PackSwq);
            if (snow->pack_temp > 0.) {
                snow->pack_temp = 0.;
            }
        }
        else {
            snow->pack_temp = 0.0;
        }
    }
    else {
        /* cold content has been either exactly satisfied or exceeded. If
           PackCC = refreeze then pack is ripe and all pack water is
           refrozen, else if energy released in refreezing exceeds PackCC
           then exactly the right amount of water is refrozen to satify PackCC.
           The refrozen water is added to PackSwq and Ice */

        snow->pack_temp = 0.0;
        DeltaPackSwq = -PackCC / (CONST_LATICE * CONST_RHOFW);
        snow->pack_water -= DeltaPackSwq;
        PackSwq += DeltaPackSwq;
        Ice += DeltaPackSwq;
    }

    /* Update the liquid water content of the pack */

    MaxLiquidWater = param.SNOW_LIQUID_WATER_CAPACITY * PackSwq;
    if (snow->pack_water > MaxLiquidWater) {
        melt[0] = snow->pack_water - MaxLiquidWater;
        snow->pack_water = MaxLiquidWater;
    }
    else {
        melt[0] = 0.0;
    }

    /* Update snow properties */

    Ice = PackSwq + SurfaceSwq;

    if (Ice > param.SNOW_MAX_SURFACE_SWE) {
        SurfaceCC = CONST_VCPICE_WQ * snow->surf_temp * SurfaceSwq;
        PackCC = CONST_VCPICE_WQ * snow->pack_temp * PackSwq;
        if (SurfaceSwq > param.SNOW_MAX_SURFACE_SWE) {
            PackCC += SurfaceCC *
                      (SurfaceSwq - param.SNOW_MAX_SURFACE_SWE) / SurfaceSwq;
            SurfaceCC -= SurfaceCC *
                         (SurfaceSwq - param.SNOW_MAX_SURFACE_SWE) / SurfaceSwq;
            PackSwq += SurfaceSwq - param.SNOW_MAX_SURFACE_SWE;
            SurfaceSwq -= SurfaceSwq - param.SNOW_MAX_SURFACE_SWE;
        }
        else if (SurfaceSwq < param.SNOW_MAX_SURFACE_SWE) {
            PackCC -= PackCC *
                      (param.SNOW_MAX_SURFACE_SWE - SurfaceSwq) / PackSwq;
            SurfaceCC += PackCC *
                         (param.SNOW_MAX_SURFACE_SWE - SurfaceSwq) / PackSwq;
            PackSwq -= param.SNOW_MAX_SURFACE_SWE - SurfaceSwq;
            SurfaceSwq += param.SNOW_MAX_SURFACE_SWE - SurfaceSwq;
        }
        snow->pack_temp = PackCC / (CONST_VCPICE_WQ * PackSwq);
        snow->surf_temp = SurfaceCC / (CONST_VCPICE_WQ * SurfaceSwq);
    }
    else {
        PackSwq = 0.0;
        PackCC = 0.0;
        snow->pack_temp = 0.0;
    }

    snow->swq = Ice + snow->pack_water + snow->surf_water;

    if (snow->swq == 0.0) {
        snow->surf_temp = 0.0;
        snow->pack_temp = 0.0;
    }

    /* Mass balance test */

    MassBalanceError = (InitialSwq - snow->swq) + (RainFall + SnowFall) -
                       melt[0] + snow->vapor_flux;

    melt[0] *= MM_PER_M;               /* converts back to mm */
    snow->mass_error = MassBalanceError;
    snow->coldcontent = SurfaceCC;
    snow->vapor_flux *= -1.;
    *save_advection = advection;
    *save_deltaCC = deltaCC;
    *save_grnd_flux = grnd_flux;
    *save_latent = latent_heat;
    *save_latent_sub = latent_heat_sub;
    *save_sensible = sensible_heat;
    *save_advected_sensible = advected_sensible_heat;
    *save_refreeze_energy = RefreezeEnergy;
    *save_Qnet = Qnet;

    return (0);
}

/******************************************************************************
 * @brief    Dummy function to make a direct call to SnowEnergyBalance()
 *           possible.
 *****************************************************************************/
double
CalcSnowPackEnergyBalance(double Tsurf,
                          ...)
{
    va_list ap;                 /* Used in traversing variable argument list
                                 */
    double  Qnet;                /* Net energy exchange at the SnowPack snow
                                    surface (W/m^2) */

    va_start(ap, Tsurf);
    Qnet = SnowPackEnergyBalance(Tsurf, ap);
    va_end(ap);

    return Qnet;
}

/******************************************************************************
 * @brief    Pass snow pack energy balance terms to
 *           ErrorPrintSnowPackEnergyBalance
 *****************************************************************************/
int
ErrorSnowPackEnergyBalance(double Tsurf,
                           ...)
{
    va_list ap;                 /* Used in traversing variable argument list
                                 */
    int     error;             /* error from ErrorPrintSnowPackEnergyBalance */

    va_start(ap, Tsurf);
    error = ErrorPrintSnowPackEnergyBalance(Tsurf, ap);
    va_end(ap);

    return (error);
}

/******************************************************************************
 * @brief    Print snow pack energy balance terms
 *****************************************************************************/
int
ErrorPrintSnowPackEnergyBalance(double  TSurf,
                                va_list ap)
{
    /* Define Variable Argument List */

    /* General Model Parameters */
    int    iveg, band;
    double Dt;                    /* Model time step (sec) */

    /* Vegetation Parameters */
    double Ra;                    /* Aerodynamic resistance (s/m) */
    double Z;                     /* Reference height (m) */
    double Z0;                    /* surface roughness height (m) */

    /* Atmospheric Forcing Variables */
    double AirDens;               /* Density of air (kg/m3) */
    double EactAir;               /* Actual vapor pressure of air (Pa) */
    double LongSnowIn;            /* Incoming longwave radiation (W/m2) */
    double Lv;                    /* Latent heat of vaporization (J/kg3) */
    double Press;                 /* Air pressure (Pa) */
    double Rain;                  /* Rain fall (m/timestep) */
    double ShortRad;              /* Net incident shortwave radiation
                                     (W/m2) */
    double Vpd;                   /* Vapor pressure deficit (Pa) */
    double Wind;                  /* Wind speed (m/s) */

    /* Snowpack Variables */
    double OldTSurf;              /* Surface temperature during previous time
                                     step */
    double SnowCoverFract;        /* Fraction of area covered by snow */
    double SnowDensity;           /* Density of snowpack (kg/m^3) */
    double SurfaceLiquidWater;    /* Liquid water in the surface layer (m) */
    double SweSurfaceLayer;       /* Snow water equivalent in surface layer
                                     (m) */

    /* Energy Balance Components */
    double  Tair;                 /* Canopy surface temperature (C) */
    double  TGrnd;                /* Ground surface temperature (C) */

    double *AdvectedEnergy;       /* Energy advected by precipitation (W/m2) */
    double *AdvectedSensibleHeat; /* Sensible heat advected from snow-free
                                     area into snow covered area (W/m^2) */
    double *DeltaColdContent;     /* Change in cold content of surface
                                     layer (W/m2) */
    double *DeltaPackColdContent; /* Change in sold content of pack
                                     layer (W/m^2) */
    double *GroundFlux;           /* Ground Heat Flux (W/m2) */
    double *LatentHeat;           /* Latent heat exchange at surface (W/m2) */
    double *LatentHeatSub;        /* Latent heat of sub exchange at
                                     surface (W/m2) */
    double *NetLongSnow;          /* Net longwave radiation at snowpack
                                     surface (W/m^2) */
    double *RefreezeEnergy;       /* Refreeze energy (W/m2) */
    double *SensibleHeat;         /* Sensible heat exchange at surface
                                     (W/m2) */
    double *VaporMassFlux;        /* Mass flux of water vapor to or from the
                                     intercepted snow */
    double *BlowingMassFlux;        /* Mass flux of water vapor to or from the
                                       intercepted snow */
    double *SurfaceMassFlux;        /* Mass flux of water vapor to or from the
                                         intercepted snow */

    /* Read Variable Argument List */

    /* General Model Parameters */
    iveg = (int) va_arg(ap, int);
    band = (int) va_arg(ap, int);
    Dt = (double) va_arg(ap, double);

    /* Vegetation Parameters */
    Ra = (double) va_arg(ap, double);
    Z = (double) va_arg(ap, double);
    Z0 = (double) va_arg(ap, double);

    /* Atmospheric Forcing Variables */
    AirDens = (double) va_arg(ap, double);
    EactAir = (double) va_arg(ap, double);
    LongSnowIn = (double) va_arg(ap, double);
    Lv = (double) va_arg(ap, double);
    Press = (double) va_arg(ap, double);
    Rain = (double) va_arg(ap, double);
    ShortRad = (double) va_arg(ap, double);
    Vpd = (double) va_arg(ap, double);
    Wind = (double) va_arg(ap, double);

    /* Snowpack Variables */
    OldTSurf = (double) va_arg(ap, double);
    SnowCoverFract = (double) va_arg(ap, double);
    SnowDensity = (double) va_arg(ap, double);
    SurfaceLiquidWater = (double) va_arg(ap, double);
    SweSurfaceLayer = (double) va_arg(ap, double);

    /* Energy Balance Components */
    Tair = (double) va_arg(ap, double);
    TGrnd = (double) va_arg(ap, double);

    AdvectedEnergy = (double *) va_arg(ap, double *);
    AdvectedSensibleHeat = (double *)va_arg(ap, double *);
    DeltaColdContent = (double *) va_arg(ap, double *);
    DeltaPackColdContent = (double *) va_arg(ap, double *);
    GroundFlux = (double *) va_arg(ap, double *);
    LatentHeat = (double *) va_arg(ap, double *);
    LatentHeatSub = (double *) va_arg(ap, double *);
    NetLongSnow = (double *) va_arg(ap, double *);
    RefreezeEnergy = (double *) va_arg(ap, double *);
    SensibleHeat = (double *) va_arg(ap, double *);
    VaporMassFlux = (double *) va_arg(ap, double *);
    BlowingMassFlux = (double *) va_arg(ap, double *);
    SurfaceMassFlux = (double *) va_arg(ap, double *);

    /* print variables */
    log_warn("snow_melt failed to converge to a solution in "
             "root_brent.  Variable values will be dumped to the "
             "screen, check for invalid values.");

    /* general model terms */
    fprintf(LOG_DEST, "iveg = %i\n", iveg);
    fprintf(LOG_DEST, "band = %i\n", band);
    fprintf(LOG_DEST, "Dt = %f\n", Dt);

    /* land surface parameters */
    fprintf(LOG_DEST, "Ra = %f\n", Ra);
    fprintf(LOG_DEST, "Z = %f\n", Z);
    fprintf(LOG_DEST, "Z0 = %f\n", Z0);

    /* meteorological terms */
    fprintf(LOG_DEST, "AirDens = %f\n", AirDens);
    fprintf(LOG_DEST, "EactAir = %f\n", EactAir);
    fprintf(LOG_DEST, "LongSnowIn = %f\n", LongSnowIn);
    fprintf(LOG_DEST, "Lv = %f\n", Lv);
    fprintf(LOG_DEST, "Press = %f\n", Press);
    fprintf(LOG_DEST, "Rain = %f\n", Rain);
    fprintf(LOG_DEST, "ShortRad = %f\n", ShortRad);
    fprintf(LOG_DEST, "Vpd = %f\n", Vpd);
    fprintf(LOG_DEST, "Wind = %f\n", Wind);

    /* snow pack terms */
    fprintf(LOG_DEST, "TSurf = %f\n", TSurf);
    fprintf(LOG_DEST, "OldTSurf = %f\n", OldTSurf);
    fprintf(LOG_DEST, "SnowCoverFract = %f\n", SnowCoverFract);
    fprintf(LOG_DEST, "SnowDensity = %f\n", SnowDensity);
    fprintf(LOG_DEST, "SurfaceLiquidWater = %f\n", SurfaceLiquidWater);
    fprintf(LOG_DEST, "SweSurfaceLayer = %f\n", SweSurfaceLayer);
    fprintf(LOG_DEST, "Tair = %f\n", Tair);
    fprintf(LOG_DEST, "TGrnd = %f\n", TGrnd);
    fprintf(LOG_DEST, "AdvectedEnergy = %f\n", AdvectedEnergy[0]);
    fprintf(LOG_DEST, "AdvectedSensibleHeat = %f\n", AdvectedSensibleHeat[0]);
    fprintf(LOG_DEST, "DeltaColdContent = %f\n", DeltaColdContent[0]);
    fprintf(LOG_DEST, "DeltaPackColdContent = %f\n", DeltaPackColdContent[0]);
    fprintf(LOG_DEST, "GroundFlux = %f\n", GroundFlux[0]);
    fprintf(LOG_DEST, "LatentHeat = %f\n", LatentHeat[0]);
    fprintf(LOG_DEST, "LatentHeatSub = %f\n", LatentHeatSub[0]);
    fprintf(LOG_DEST, "NetLongSnow = %f\n", NetLongSnow[0]);
    fprintf(LOG_DEST, "RefreezeEnergy = %f\n", RefreezeEnergy[0]);
    fprintf(LOG_DEST, "SensibleHeat = %f\n", SensibleHeat[0]);
    fprintf(LOG_DEST, "VaporMassFlux = %f\n", VaporMassFlux[0]);
    fprintf(LOG_DEST, "BlowingMassFlux = %f\n", BlowingMassFlux[0]);
    fprintf(LOG_DEST, "SurfaceMassFlux = %f\n", SurfaceMassFlux[0]);

    log_warn("Finished dumping snow_melt variables.\nTry increasing "
             "SNOW_DT to get model to complete cell.\nThencheck output "
             "for instabilities.");

    return(ERROR);
}
