% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RVineStdError.R
\name{RVineStdError}
\alias{RVineStdError}
\title{Standard Errors of an R-Vine Copula Model}
\usage{
RVineStdError(hessian, RVM)
}
\arguments{
\item{hessian}{The Hessian matrix of the given R-vine.}

\item{RVM}{An \code{\link[=RVineMatrix]{RVineMatrix()}} object including the structure, the
pair-copula families, and the parameters.}
}
\value{
\item{se}{The calculated standard errors for the first parameter
matrix. The entries are ordered with respect to the ordering of the
\code{RVM$par} matrix.} \item{se2}{The calculated standard errors for the
second parameter matrix.}
}
\description{
This function calculates the standard errors of a d-dimensional R-vine
copula model given the Hessian matrix.
}
\note{
The negative Hessian matrix should be positive semidefinite. Otherwise
NAs will be returned in some entries and the non-NA entries may be wrong. If
the negative Hessian matrix is negative definite, then one could try a near
positive matrix. The package \code{Matrix} provides a function called
\code{nearPD} to estimate a matrix which is positive definite and close to
the given matrix.
}
\examples{

# define 5-dimensional R-vine tree structure matrix
Matrix <- c(5, 2, 3, 1, 4,
            0, 2, 3, 4, 1,
            0, 0, 3, 4, 1,
            0, 0, 0, 4, 1,
            0, 0, 0, 0, 1)
Matrix <- matrix(Matrix, 5, 5)

# define R-vine pair-copula family matrix
family <- c(0, 1, 3, 4, 4,
            0, 0, 3, 4, 1,
            0, 0, 0, 4, 1,
            0, 0, 0, 0, 3,
            0, 0, 0, 0, 0)
family <- matrix(family, 5, 5)

# define R-vine pair-copula parameter matrix
par <- c(0, 0.2, 0.9, 1.5, 3.9,
         0, 0, 1.1, 1.6, 0.9,
         0, 0, 0, 1.9, 0.5,
         0, 0, 0, 0, 4.8,
         0, 0, 0, 0, 0)
par <- matrix(par, 5, 5)

# define second R-vine pair-copula parameter matrix
par2 <- matrix(0, 5, 5)

# define RVineMatrix object
RVM <- RVineMatrix(Matrix = Matrix, family = family,
                   par = par, par2 = par2,
                   names = c("V1", "V2", "V3", "V4", "V5"))

# simulate a sample of size 300 from the R-vine copula model
set.seed(123)
simdata <- RVineSim(300, RVM)

# compute the Hessian matrix of the first row of the data
out2 <- RVineHessian(simdata,RVM)

# get the standard errors
RVineStdError(out2$hessian, RVM)

}
\references{
Dissmann, J. F., E. C. Brechmann, C. Czado, and D. Kurowicka
(2013). Selecting and estimating regular vine copulae and application to
financial returns. Computational Statistics & Data Analysis, 59 (1), 52-69.

Schepsmeier, U. and J. Stoeber (2014)
Derivatives and Fisher information of bivariate copulas.
Statistical Papers, 55(2), 525-542.
online first: \url{https://link.springer.com/article/10.1007/s00362-013-0498-x}.

Web supplement: Derivatives and Fisher Information of bivariate copulas.
\url{https://mediatum.ub.tum.de/node?id=1119201}

Stoeber, J. and U. Schepsmeier (2013). Estimating standard errors in regular
vine copula models. Computational Statistics, 28 (6), 2679-2707
\url{https://link.springer.com/article/10.1007/s00180-013-0423-8#}.
}
\seealso{
\code{\link[=BiCopDeriv]{BiCopDeriv()}},
\code{\link[=BiCopDeriv2]{BiCopDeriv2()}},
\code{\link[=BiCopHfuncDeriv]{BiCopHfuncDeriv()}},
\code{\link[=BiCopHfuncDeriv2]{BiCopHfuncDeriv2()}}, \cr
\code{\link[=RVineMatrix]{RVineMatrix()}},
\code{\link[=RVineHessian]{RVineHessian()}},
\code{\link[=RVineGrad]{RVineGrad()}}
}
\author{
Ulf Schepsmeier, Jakob Stoeber
}
