\name{WEE.quantile}
\alias{WEE.quantile}
\title{
WEE quantile regression
}
\description{
Returns an object of class "WEE.quantile" that is generated by quantile regression with WEE approach for continuous secondary traits in genetic case-control studies.
}
\usage{
WEE.quantile(formula, D, data, pd_pop, tau, iter = 5, boot = 0, ...)
}
\arguments{
  \item{formula}{
The secondary trait given SNPs and covariates.  e.g. y~x+z
}
  \item{D}{
Primary disease (case-control status), must be specified. 
}
  \item{data}{
Dataset with real observation.	
}
  \item{pd_pop}{
The population disease prevelance of primary disease.
}
  \item{tau}{
The quantile level to be estimated. Multiple taus can be chosen.
}
  \item{iter}{
Number of generating pseudo observations. (iter=10 by default)
}
  \item{boot}{
Number of bootstrape samples. (boot=0 by default)	
}
\item{\dots}{
Optional arguments to be passed through to rq. 
}
}
\details{
The quantile regression package "quantreg" is required before calling this function
}
\value{
\item{Coefficients}{Point estimates}
\item{StdErr}{Bootstrap standard errors, returned if boot > 0}
\item{Wald}{Wald test statistics, returned if boot > 0}
\item{p.value}{p-values, returned if boot > 0}
\item{Covariance}{Covariance matrix, returned if boot > 0}
}
\section{Warning}{
If boot = 0, point estimates are plotted. If boot > 0, boostrap standard errors, Wald test statistics, p-values, and covariance matrix are also returned. 
Optional arguments from rq can be passed to this function, but arguments 'subset' and 'weights' should be used with caution. 
}
\examples{
## Generate simulated data
# set population size as 500000
n = 500000 

# set parameters
beta = c(0.12, 0.1) # P(Y|X,Z)
gamma = c(-4, log(1.5), log(1.5), log(2) ) #P(D|X,Y,Z)

# generate the genetic variant X
x = rbinom(n,size=2,prob=0.3)
  
# generate the continuous covariate Z
z = rnorm(n)
  
# generate the continuous secondary trait Y
y= 1 + beta[1]*x + beta[2]*z + (1+0.02*x)*rnorm(n)

# generate disease status D
p = exp(gamma[1]+x*gamma[2]+z*gamma[3]+y*gamma[4])/
(1+exp(gamma[1]+x*gamma[2]+z*gamma[3]+y*gamma[4]))
d = rbinom(n,size=1,prob=p)

# form population data dataset
dat = as.data.frame(cbind(x,y,z,d))
colnames(dat) = c("x","y","z","d")

# Generate sample dataset with 200 cases and 200 controls
dat_cases = dat[which(dat$d==1),]
dat_controls= dat[which(dat$d==0),]
dat_cases_sample = dat_cases[sample(sum(dat$d==1),
                             200, replace=FALSE),]
dat_controls_sample = dat_controls[sample(sum(dat$d==0),
                                   200, replace=FALSE),]

dat_quantile = as.data.frame(rbind(dat_cases_sample,
                                   dat_controls_sample))
colnames(dat_quantile) = c("x","y","z","D")
D = dat_quantile$D # Disease status
pd = sum(d==1)/n # population disease prevalence

# WEE quantile regressions:
WEE.quantile(y ~ x, D, tau = 0.5, 
             data = dat_quantile, pd_pop = pd)	  				
\donttest{
WEE.quantile(y ~ x + z, D, tau = 1:9/10, 
             data = dat_quantile, pd_pop = pd, boot = 500)
}
}

\references{
[1] Ying Wei, Xiaoyu Song, Mengling Liu, Iuliana Ionita-Laza and Joan Reibman
(2016). Quantile Regression in the Secondary Analysis of Case Control Data. \emph{Journal of the American Statistical Association}, 111:513, 344-354; DOI: 10.1080/01621459.2015.1008101

[2] Xiaoyu Song, Iuliana Ionita-Laza, Mengling Liu, Joan Reibman, Ying Wei (2016). A General and Robust Framework for Secondary Traits Analysis. \emph{Genetics}, vol. 202 no. 4 1329-1343; DOI: 10.1534/genetics.115.181073
}



