#'@title Retrieves categories associated with a page.
#'
#'@description
#'Retrieves categories associated with a page (or list of pages) on a MediaWiki instance
#'
#'@param con A connector object, generated by \code{\link{wiki_con}}, that corresponds
#'to the project you're trying to query.
#'
#'@param pages A vector of page titles, with or without spaces, that you want to retrieve
#'categories for.
#'
#'@param properties The properties you want to retrieve about the categories.
#'Options are "sortkey" (the key that sorts the way the page is stored in each category),
#'"timestamp" (when the category was added to that page) and "hidden" (tags those categories
#'in the returned list that are 'hidden' from readers).
#'
#'@param limit The maximum number of categories you want to retrieve for each page. Set
#'to 50 by default.
#'
#'@param show_hidden Whether or not to include 'hidden' categories in the categories
#'that are retrieved - these are usually associated with the maintenance of Wikipedia
#'and its internal processes. Set to FALSE by default.
#'
#'@seealso \code{\link{wiki_catpages}} for pages in a specified category
#'
#'@export

wiki_pagecats <- function(con, pages, properties = c("sortkey","timestamp","hidden"), limit = 50,
                          show_hidden = FALSE){
  
  #Normalise and save the properties
  properties <- match.arg(properties, several.ok = TRUE)
  properties <- paste(properties, collapse = "|")
  
  #Check the number of pages in the request
  pages <- LimitHandler(pages, 50)
  
  if(show_hidden){
    
    show_hidden <- "hidden"
    
  } else {
    
    show_hidden <- "!hidden"
    
  }
  
  #Construct the URL
  pagecat_url <- paste(con$URL,"&action=query&prop=categories&clprop=",properties,"&clshow=",show_hidden,"&cllimit=",limit,"&titles=",pages, sep = "")
  
  #Retrieve the content
  pagecat_content <- wiki_call(URL = pagecat_url, con$CurlOpts)
  
  #Check for invalid pageIDs
  MissingPagesHandler(pagecat_content)
  
  #Return
  return(pagecat_content)
}