% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/abcrlda.R
\name{abcrlda}
\alias{abcrlda}
\title{Asymptotically Bias-Corrected Regularized Linear Discriminant Analysis
for Cost-Sensitive Binary Classification}
\usage{
abcrlda(x, y, gamma = 1, cost = c(0.5, 0.5), bias_correction = TRUE)
}
\arguments{
\item{x}{Input matrix or data.frame of dimension \code{nobs x nvars}; each row is an feature vector.}

\item{y}{A numeric vector or factor of class labels. Factor should have either two levels or be
a vector with two distinct values.
If \code{y} is presented as a vector, it will be coerced into a factor.
Length of \code{y} has to correspond to number of samples in \code{x}.}

\item{gamma}{Regularization parameter \eqn{\gamma}{gamma} in the ABC-RLDA discriminant function given by:
\deqn{W_{ABC}^{RLDA} = \gamma (x-\frac{\bar{x}_0 +
\bar{x}_1}{2})^T H (\bar{x}_0 - \bar{x}_1)
- log(\frac{C_{01}}{C_{10}}) + \hat{\omega}_{opt}}{W_ABCRLDA = gamma (x - (x0 + x1)/2) H (x0 - x1) + log(C_01/C_10) + omega_opt}
\deqn{H = (I_p + \gamma \hat{\Sigma})^{-1}}{H = (I_p + gamma Sigma_hat)^-1}
Formulas and derivations for parameters used in above equation can be found in the article under reference section.}

\item{cost}{Parameter that controls the overall misclassification costs.
 This is a vector of length 1 or 2 where the first value is \eqn{C_{10}}{C_10} (represents the cost of assigning label 1 when the true label is 0)
 and the second value, if provided, is \eqn{C_{01}}{C_01} (represents the cost of assigning label 0 when the true label is 1).
 The default setting is c(0.5, 0.5), so both classes have equal misclassification costs

 If a single value is provided, it should be normalized to lie between 0 and 1 (but not including 0 or 1).
 This value will be assigned to \eqn{C_{10}}{C_10} while
 \eqn{C_{01}}{C_01} will be equal to \eqn{(1 - C_{10})}{1 - C_10}.}

\item{bias_correction}{Takes in a boolean value.
If \code{bias_correction} is TRUE, then asymptotic bias correction will be performed.
Otherwise, (if \code{bias_correction} is FALSE) asymptotic bias correction will not be performed and
the ABCRLDA is the classical RLDA.
The default is TRUE.}
}
\value{
An object of class "abcrlda" is returned which can be used for class prediction (see predict()).
  \item{a}{Coefficient vector of a discriminant hyperplane: W(\strong{x}) = \strong{a}' \strong{x} + m.}
  \item{m}{Intercept of discriminant hyperplane: W(\strong{x}) = \strong{a}'\strong{x} + m.}
  \item{cost}{Vector of cost values that are used to construct ABC-RLDA.}
  \item{ncost}{Normilized cost such that \eqn{C_{10}}{C_10} + \eqn{C_{01}}{C_01} = 1.}
  \item{gamma}{Regularization parameter value used in ABC_RLDA discriminant function.}
  \item{lev}{Levels corresponding to the labels in y.}
}
\description{
Constructs Asymptotically Bias-Corrected Regularized Linear Discriminant Analysis.
}
\section{Reference}{

  A. Zollanvari, M. Abdirash, A. Dadlani and B. Abibullaev,
  "Asymptotically Bias-Corrected Regularized Linear Discriminant Analysis for Cost-Sensitive
  Binary Classification," in IEEE Signal Processing Letters, vol. 26, no. 9, pp. 1300-1304,
  Sept. 2019. doi: 10.1109/LSP.2019.2918485
  URL: \url{http://ieeexplore.ieee.org/stamp/stamp.jsp?tp=&arnumber=8720003&isnumber=8770167}
}

\examples{
data(iris)
train_data <- iris[which(iris[, ncol(iris)] == "virginica" |
                           iris[, ncol(iris)] == "versicolor"), 1:4]
train_label <- factor(iris[which(iris[, ncol(iris)] == "virginica" |
                                   iris[, ncol(iris)] == "versicolor"), 5])
model <- abcrlda(train_data, train_label, gamma = 0.5, cost = 0.75)
a <- predict(model, train_data)
# same params but more explicit
model <- abcrlda(train_data, train_label, gamma = 0.5, cost = c(0.75, 0.25))
b <- predict(model, train_data)
# same class costs ratio
model <- abcrlda(train_data, train_label, gamma = 0.5, cost = c(3, 1))
c <- predict(model, train_data)
# all this model will give the same predictions
all(a == b & a == c & b == c)
#' [1] TRUE
}
\seealso{
Other functions in the package: 
\code{\link{cross_validation}()},
\code{\link{da_risk_estimator}()},
\code{\link{grid_search}()},
\code{\link{predict.abcrlda}()},
\code{\link{risk_calculate}()}
}
\concept{functions in the package}
