#' ADEA analysis to variable selection in DEA
#'
#' It does an ADEA analysis. In particular it computes a score for each DMU and a load for each variable.
#'
#' The selection of input and output variables for inclusion in a DEA model is a critical aspect, as many studies have revealed.
#' \code{adea} function provides an implementation of the ADEA method for variable selection in DEA.
#'
#' ADEA methodology introduces a new phase in classical DEA analysis, measuring the relative importance of each input and output variable.
#' This measure is referred to as load or contribution.
#' It also defines a load for the entire model.
#' Using this measure, a procedure has been developed to select an optimised or relevant set of variables.
#'
#' A variable's load is a standardized ratio of the efficiency scores of all DMUs due to that variable.
#' These loads quantify the contribution of each variable to the overall model.
#' Where 0 means that the contribution of that variable to the efficiency values is negligible.
#' The ideal load is 1, and values range from 0 to the number of input or output variables.
#' The lowest load value has a real significance as it represents the variable with the least contribution to efficiency.
#'
#' As it is usually done in DEA, these loads are computed as its maximum allowable value using alternative sets of weights but without changing the efficiency scores.
#' But because the sum of all of them is fixed, when one variable increases its load, any other decrease in value.
#' So only the lowest value of all loads has a real meaning.
#' This lowest value can be taken as a significance measure of the entire model.
#'
#' This measure, load, has two key properties that easy its interpretation:
#' \itemize{
#'    \item It has a bounded range from 0 to the number of input or output variables, with 1 as the ideal value.
#'    \item It is invariant by changes in scale.
#' }
#' 
#' ADEA analysis can be done considering only input variables, in this case ADEA analysis has input as \code{load.orientation} value.
#' output when only output variables are considered.
#' And inoutput \code{load.orientation} when all variables in the model are taken into account.
#'
#' Adea models, as the DEA models, can be input or output orientated.
#' Input orientated DEA models propose to reach the efficiency of DMUs through a reduction in the amount of input required by non efficient DMUs.
#' On the contrary, output orientated DEA models propose to increase the amount of output of non efficient DMUs.
#'
#' For a detailed description of the maths behind the model, see the references.
#' 
#' @references Stepwise Selection of Variables in DEA Using Contribution Load.
#' \emph{F. Fernandez-Palacin}, \emph{M. A. Lopez-Sanchez} and \emph{M. Munoz-Marquez}.
#' Pesquisa Operacional 38 (1), pg. 1-24, 2018.
#' <DOI:10.1590/0101-7438.2018.038.01.0000>.
#' 
#' @references Methodology for calculating critical values of relevance measures in variable selection methods in data envelopment analysis.
#' \emph{Jeyms Villanueva-Cantillo} and \emph{Manuel Munoz-Marquez}.
#' European Journal of Operational Research, 290 (2), pg. 657-670, 2021.
#' <DOI:10.1016/j.ejor.2020.08.021>.
#' #
#' @param input A matrix or a data frame containing the inputs of the units to be evaluated, with one row for each DMU and one column for each input.
#' @param output A matrix or a data frame containing the outputs of the units to be evaluated, with one row for each DMU and one column for each output.
#' @param orientation Use "input" for input orientation or "output" for output orientation in DEA model.
#' The default is "input".
#' @param load.orientation This parameter allows the selection of variables to be included in load analysis. 
#' The default is "inoutput" which means that all input and output variables will be included. Use "input" or "output" to include only input or output variables in load analysis.
#' @param name An optional descriptive name for the model.
#' The default is an empty string. 
#' This name will be displayed in printed and summarized results.
#' @param solver The solver used by ROI to solve the DEA optimization problem.
#' The default is "auto."
#' The solver must be installed and capable of solving linear programming problems. 
#' Use \code{ROI_installed_solvers()} to list them.
#' @return The function returns an adea class object with the following named members:
#' \itemize{
#' \item name: A label of the model
#' \item orientation: DEA model orientation 'input' or 'output'
#' \item load.orientation: load DEA model orientation 'input', 'output', or 'inoutput'
#' \item inputnames: Variable input names
#' \item outputnames: Variable output names
#' \item eff: is a vector with DMU's scores
#' \item loads: A list with all information about loads:
#'   \itemize{
#'     \item load: The lowest load, which is the load of the ADEA model
#'     \item input: A vector with loads of input variables
#'     \item iinput: Index of input variable that reach the load of the model
#'     \item output: A vector with loads of output variables
#'     \item ioutput: Index of output variable that reach the load of the model
#'   }
#' \item ux: A set of weights for inputs
#' \item vy: A set of weights for output
#' \item vinput: Standardized virtual input dividing by the sum of the weights, see [Costa2006] in \code{\link{adea-package}}.
#' \item voutput: Standardized virtual output dividing by the sum of the weights, see [Costa2006] in \code{\link{adea-package}}
#' \item solver: The solver used for the resolution of the optimization problem
#' }
#' @seealso \code{\link{adea-package}}.
#' @examples
#' # Load data
#' data('cardealers4')
#' 
#' # Define input and output
#' input <- cardealers4[, c('Employees', 'Depreciation')]
#' output <- cardealers4[, c('CarsSold', 'WorkOrders')]
#' 
#' # Compute adea model
#' model <- adea(input, output, name = 'ADEA for cardealers4 dataset')
#' model
#' # Dealer A  Dealer B  Dealer C  Dealer D  Dealer E  Dealer F
#' # 0.9915929 1.0000000 0.8928571 0.8653846 1.0000000 0.6515044
#' 
#' # Get model's load
#' model$loads$load
#' # [1] 0.6666667
#' 
#' # Get variable loads
#' model$loads
#' # $load
#' # [1] 0.6666667
#' # $input
#' # Employees Depreciation
#' # 0.6666667    1.3333333
#' # $iinput
#' # Employees 
#' #         1 
#' # $output
#' # CarsSold WorkOrders
#' # 1.2663476  0.7336524 
#' # $ioutput
#' # WorkOrders 
#' #          2 
#'
#' # Summarize the model and print additional information
#' summary(model)
#' # Model name              ADEA for cardealers4 dataset
#' # Orientation                                    input
#' # Load orientation                            inoutput
#' # Model load                         0.666666666666659
#' # Input load.Employees               0.666666666666659
#' # Input load.Depreciation             1.33333333333334
#' # Output load.CarsSold                 1.1025075271907
#' # Output load.WorkOrders               0.8974924728093
#' # Inputs                        Employees Depreciation
#' # Outputs                          CarsSold WorkOrders
#' # nInputs                                            2
#' # nOutputs                                           2
#' # nVariables                                         4
#' # nEfficients                                        2
#' # Eff. Mean                           0.90022318389575
#' # Eff. sd                            0.135194867030839
#' # Eff. Min.                          0.651504424778761
#' # Eff. 1st Qu.                       0.872252747252747
#' # Eff. Median                        0.942225031605562
#' # Eff. 3rd Qu.                       0.997898230088495
#' # Eff. Max.                                          1
#' @export
adea <- function(input, output, orientation = c('input', 'output'), load.orientation = c('inoutput', 'input', 'output'), name = '', solver = 'auto')
{
    ## Check input and output
    err <- adea_check(input = input, output = output, ux = NULL, vy = NULL, eff = NULL)
    if (!isTRUE(err)) stop(err)

    ## Check other parameters
    orientation <- match.arg(orientation)
    load.orientation <- match.arg(load.orientation)

    ## Canonize input and output format
    data <- adea_setup(input, output)
    input <- data$input
    output <- data$output

    ## Call to low level function
    adea_(input = input, output = output, orientation = orientation, load.orientation = load.orientation, name = name, solver = solver)
}

## Same as adea but without exhaustive input check, for internal use only
adea_ <- function(input, output, orientation, load.orientation, name, solver)
{
    ## Solve adea model
    .adea <- roi_adea(input = input, output = output, orientation = orientation, load.orientation = load.orientation, solver = solver)

    ## Compute and store virtual input and output as described in @references A new approach to the bi-dimensional representation of the DEA efficient frontier with multiple inputs and outputs
    virtuals <- virtual_dea(input = input, output = output, orientation = orientation, ux = .adea$ux, vy = .adea$vy)

    .adea <- list('name' = name,
                  'orientation' = orientation,
                  'load.orientation' = load.orientation,
                  'inputnames' = colnames(input),
                  'outputnames' = colnames(output),
                  'eff' = .adea$eff,
                  'loads' = .adea$loads,
                  'ux' = .adea$ux,
                  'vy' = .adea$vy,
                  'vinput' = virtuals$vinput,
                  'voutput' = virtuals$voutput,
                  'solver' = .adea$solver)

    ## Set up the class of the object
    class(.adea) <- 'adea'
    .adea
}
