#' ADEA plot of standarized virtual input and virtual output
#'
#' @aliases plot plot,adea-method
#' @docType methods
#' @keywords plot
#'
#' @param x ADEA object to be plotted
#' @param main if not null then it is used as main in plot.
#' Its default value is the translation of "ADEA efficient frontier".
#' If the adea x has name it is added to the previous value.
#' @param xlab if not null then it is used as xlab in plot.
#' Its default value is the translation of "Virtual input".
#' @param ylab if not null then it is used as ylab in plot.
#' Its default value is the translation of "Virtual output".
#' @param labels if not null then a vector of labels for the DMUs points
#' @param labels.pos position for the labels in the plot. Its default value is 4.
#' @param lcol the color to use to draw the line.
#' Its default value is black.
#' @param ... Adittional parameters to plot
#' @return A list with vinput and voutput values.
#' These values are provided mainly for use with the function identify.
#'
#' @details
#' This function plots virtual input and virtual outpus in an ADEA model.
#' The virtual input and output vectors are computed as a weighted sum of the inputs and outputs.
#' In addition, it is imposed that the sum of the weights be the unit.
#'
#' For more information on this calculation process see the references in \cite{adea-package}.
#' 
#' For the calculations of virtual input and virtual output, the weights generated by ADEA have been used, but they are the same as those that would be obtained using standard DEA.
#' 
#' @examples
#' data("cardealers4")
#' input = cardealers4[, c('Employees', 'Depreciation')]
#' output = cardealers4[, c('CarsSold', 'WorkOrders')]
#' adea <- adea(input = input, output = output)
#' plot(adea)
#' @seealso \code{\link{adea}}
#' @importFrom graphics abline
#' @importFrom graphics text
#' @export
plot.adea <- function(x, main = NULL, xlab = NULL, ylab= NULL, labels = NULL, labels.pos = 4, lcol = "black", ...) {
    if (is.null(main)) main <- gettext("DEA efficient frontier")
    if (x$name != "") main <- paste0(main, " ", gettext("for model"), ": ", x$name)
    if (is.null(xlab)) xlab <- gettext("Virtual input")
    if (is.null(ylab)) ylab <- gettext("Virtual output")
    plot(x$vinput, x$voutput, main = main, xlab = xlab, ylab = ylab, ...)
    abline(a = 0, b = 1, col = lcol)
    if (!is.null(labels)) text(x$vinput, x$voutput, labels = labels, pos = labels.pos)
    invisible(list(vinput = x$vinput, voutput = x$voutput))
}
