% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/segmentPattern.R
\name{segmentPattern}
\alias{segmentPattern}
\title{Pattern Segmentation From a Time-series via ADEPT}
\usage{
segmentPattern(x, x.fs, template, pattern.dur.seq,
  similarity.measure = "cov", similarity.measure.thresh = 0,
  x.adept.ma.W = NULL, finetune = NULL, finetune.maxima.ma.W = NULL,
  finetune.maxima.nbh.W = NULL, run.parallel = FALSE,
  run.parallel.cores = NULL, x.cut = TRUE, x.cut.vl = 6000,
  compute.template.idx = FALSE)
}
\arguments{
\item{x}{A numeric vector. A time-series to segment pattern from.}

\item{x.fs}{A numeric scalar. Frequency at which a time-series \code{x} is collected,
expressed in a number of observations per second.}

\item{template}{A list of numeric vectors, or a numeric vector.
Each vector represents a distinct pattern template used in segmentation.}

\item{pattern.dur.seq}{A numeric vector. A grid of potential pattern durations
used in segmentation. Expressed in seconds. See: Details.}

\item{similarity.measure}{A character scalar. Statistic used to compute similarity
between a time-series \code{x}  and pattern templates. Currently supported values:
\itemize{
  \item \code{"cov"} - covariance,
  \item \code{"cor"} - correlation,
}
Default is \code{"cov"}.}

\item{similarity.measure.thresh}{A numeric scalar. Threshold of minimal similarity
value between a time-series \code{x} and a template
below which the algorithm does not identify a pattern occurrence from \code{x}.
Default is \code{0}.}

\item{x.adept.ma.W}{A numeric scalar.
A length of a window used in moving average smoothing of a time-series \code{x} for
  similarity matrix computation. Expressed in seconds.
 Default is \code{NULL} (no smoothing applied).}

\item{finetune}{A character scalar. A type of fine-tuning procedure employed in
 segmentation. Defaults to \code{NULL} (no fine-tuning procedure employed). Currently supported values:
\itemize{
  \item \code{"maxima"} - tunes preliminarily identified
  beginning and end of a pattern so
  as they correspond to local maxima of time-series \code{x} (or smoothed version of \code{x})
  found within neighbourhoods of preliminary locations.
}}

\item{finetune.maxima.ma.W}{A numeric scalar.
A length of a window used in moving average smoothing of a time-series \code{x} in
 \code{"maxima"} fine-tuning procedure. Expressed in seconds.
 Default is \code{NULL} (no smoothing applied).}

\item{finetune.maxima.nbh.W}{A numeric scalar.
A length of the two neighborhoods centered at preliminarily identified
beginning and end of a pattern
within which we search for local maxima of \code{x} (or smoothed version of \code{x}) in \code{"maxima"}
fine-tuning procedure. Expressed in seconds. Default is \code{NULL}.
Note: if the length provided corresponds to an even number of \code{x} vector indices,
it will be rounded down so as the corresponding number of vector indices is its closest odd number.}

\item{run.parallel}{A logical scalar. Whether or not to use parallel execution in the algorithm.
The \code{future} package
is used to execute code asynchronously. Default is \code{FALSE}.}

\item{run.parallel.cores}{An integer scalar.
The number of cores to use for parallel execution.
Default is \code{NULL}. If not specified, the number of cores is set to a number of
cores available minus 1.}

\item{x.cut}{A logical scalar. Whether or not to use time optimization procedure in
which a time-series \code{x} is cut into parts and segmentation is performed for
each part of \code{x} separately. Recommended for a time-series \code{x} of vector length
 above 30,000. Default is \code{TRUE}.}

\item{x.cut.vl}{An integer scalar.
Defines a vector length of parts that \code{x} vector is cut into during the execution time optimization procedure.
Default is \code{6000} (recommended).}

\item{compute.template.idx}{A logical scalar. Whether or not to compute and return information about
which of the provided pattern templates yielded a similarity matrix value
that corresponds to an identified pattern occurrence.
Setting to \code{TRUE} may increase computation time. Default is \code{FALSE}.}
}
\value{
A \code{data.frame} with segmentation results. Each row
describes one identified pattern occurrence:
\itemize{
  \item \code{tau_i} - index of  \code{x} where pattern starts,
  \item \code{T_i} - pattern duration, expressed in \code{x} vector length,
  \item \code{sim_i} -  similarity between a pattern and \code{x};
  note: if \code{"maxima"} fine-tune and/or \code{x} smoothing is employed,
  the similarity value between the final segmented pattern and a template
   may differ from the value in this table,
  \item \code{template_i} - if \code{compute.template.idx} equals \code{TRUE}:
  index of a template best matched to \code{x};
 if \code{compute.template.idx} equals \code{FALSE}: \code{NA}.
}
}
\description{
Segment pattern from a time-series \code{x} via Adaptive Empirical Pattern
Transformation (ADEPT).
}
\details{
Function implements Adaptive Empirical Pattern Transformation (ADEPT) method for pattern segmentation
from a time-series \code{x}.
ADEPT is optimized to perform fast, accurate walking strides segmentation from
high-density data collected with a wearable accelerometer during walking.

 ADEPT identifies patterns in a time-series \code{x} via maximization of chosen
 similarity statistic (correlation, covariance, etc.) between a time-series \code{x}
 and a pattern template(s). It accounts for variability in both
 (1) pattern duration and (2) pattern shape.
}
\examples{
## Example 1: Simulate a time-series `x`. Assume that
## - `x` is collected at a frequency of 100 Hz,
## - there is one shape of pattern present within `x`,
## - each pattern lasts 1 second,
## - there is no noise in the collected data.
true.pattern <- cos(seq(0, 2 * pi, length.out = 100))
x <- c(true.pattern[1], replicate(10, true.pattern[-1]))
## Segment pattern from x.
out <- segmentPattern(
  x = x,
  x.fs = 100,
  template = true.pattern,
  pattern.dur.seq = c(0.9, 0.95, 1.03, 1.1),
  similarity.measure = "cor",
  compute.template.idx = TRUE)
out
## Segment pattern from x. Now assume a grid of potential pattern duratios
## contains true pattern duration
out <- segmentPattern(
  x = x,
  x.fs = 100,
  template = true.pattern,
  pattern.dur.seq = c(0.9, 0.95, 1, 1.03, 1.1),
  similarity.measure = "cor",
  compute.template.idx = TRUE)
out

## Example 2: Simulate a time-series `x`. Assume that
## - `x` is collected at a frequency of 100 Hz,
## - there are two shapes of pattern present within `x`,
## - patterns have various duration,
## - there is no noise in the collected data.
true.pattern.1 <- cos(seq(0, 2 * pi, length.out = 200))
true.pattern.2 <- true.pattern.1
true.pattern.2[70:130] <- 2 * true.pattern.2[min(70:130)] + abs(true.pattern.2[70:130])
x <- numeric()
for (vl in seq(70, 130, by = 10)){
  true.pattern.1.s <- approx(
    seq(0, 1, length.out = 200),
    true.pattern.1, xout = seq(0, 1, length.out = vl))$y
  true.pattern.2.s <- approx(
    seq(0, 1, length.out = 200),
    true.pattern.2, xout = seq(0, 1, length.out = vl))$y
  x <- c(x, true.pattern.1.s[-1], true.pattern.2.s[-1])
  if (vl == 70) x <- c(true.pattern.1.s[1], x)
}
## Segment pattern from x. Use a `template` object consisting of both
## true patterns used in `x` simulation.
out <- segmentPattern(
  x = x,
  x.fs = 100,
  template = list(true.pattern.1, true.pattern.2),
  pattern.dur.seq = 60:130 * 0.01,
  similarity.measure = "cor",
  compute.template.idx = TRUE)
out

## Example 3: Simulate a time-series `x`. Assume that
## - `x` is collected at a frequency of 100 Hz,
## - there are two shapes of a pattern present within `x`,
## - patterns have various duration,
## - there is noise in the collected data.
set.seed(1)
x <- x + rnorm(length(x), sd = 0.5)
## Segment pattern from x.
out <- segmentPattern(
  x = x,
  x.fs = 100,
  template = list(true.pattern.1, true.pattern.2),
  pattern.dur.seq =  60:130 * 0.01,
  similarity.measure = "cor",
  compute.template.idx = TRUE)
out
## Segment pattern from x. Use `x.adept.ma.W` to define a length of a smoothing
## window to smooth `x` for similarity matrix computation.
out <- segmentPattern(
  x = x,
  x.fs = 100,
  template = list(true.pattern.1, true.pattern.2),
  pattern.dur.seq =  60:130 * 0.01,
  similarity.measure = "cor",
  x.adept.ma.W = 0.1,
  compute.template.idx = TRUE)
out
## Segment pattern from x. Use `x.adept.ma.W` to define a length of a smoothing
## window to smooth `x` for similarity matrix computation. Employ a fine-tuning
## procedure for stride identification.
out <- segmentPattern(
  x = x,
  x.fs = 100,
  template = list(true.pattern.1, true.pattern.2),
  pattern.dur.seq =  60:130 * 0.01,
  similarity.measure = "cor",
  x.adept.ma.W = 0.1,
  finetune = "maxima",
  finetune.maxima.nbh.W = 0.3,
  compute.template.idx = TRUE)
out
## Segment pattern from x. Employ a fine-tuning procedure for stride
## identification. Smooth `x` for both similarity matrix computation
## (set `x.adept.ma.W = 0.1`) and for  fine-tune peak detection procedure
## (set `finetune.maxima.nbh.W = 0.3`).
out <- segmentPattern(
  x = x,
  x.fs = 100,
  template = list(true.pattern.1, true.pattern.2),
  pattern.dur.seq =  60:130 * 0.01,
  similarity.measure = "cor",
  x.adept.ma.W = 0.1,
  finetune = "maxima",
  finetune.maxima.nbh.W = 0.3,
  compute.template.idx = TRUE)
out

}
