% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/put_object.R
\name{put_object}
\alias{put_object}
\alias{put_folder}
\title{Put object}
\usage{
put_object(file, object, bucket, multipart = FALSE, acl = c("private",
  "public-read", "public-read-write", "aws-exec-read", "authenticated-read",
  "bucket-owner-read", "bucket-owner-full-control"), headers = list(), ...)

put_folder(folder, bucket, ...)
}
\arguments{
\item{file}{A character string containing the filename (or full path) of the file you want to upload to S3. Alternatively, an raw vector containing the file can be passed directly, in which case \code{object} needs to be specified explicitly.}

\item{object}{A character string containing the name the object should have in S3 (i.e., its "object key"). If missing, the filename is used.}

\item{bucket}{Character string with the name of the bucket, or an object of class \dQuote{s3_bucket}.}

\item{multipart}{A logical indicating whether to use multipart uploads. See \url{http://docs.aws.amazon.com/AmazonS3/latest/dev/mpuoverview.html}. If \code{file} is less than 100 MB, this is ignored.}

\item{acl}{A character string indicating a \href{http://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl}{\dQuote{canned} access control list}. By default all bucket contents and objects therein are given the ACL \dQuote{private}. This can later be viewed using \code{\link{get_acl}} and modified using \code{\link{put_acl}}.}

\item{headers}{List of request headers for the REST call.}

\item{\dots}{Additional arguments passed to \code{\link{s3HTTP}}.}

\item{folder}{A character string containing a folder name. (A trailing slash is not required.)}
}
\value{
If successful, \code{TRUE}.
}
\description{
Puts an object into an S3 bucket
}
\details{
This provide a generic interface for sending files (or serialized, in-memory representations thereof) to S3. Some convenience wrappers are provided for common tasks: e.g., \code{\link{s3save}} and \code{\link{s3saveRDS}}.

Note that S3 is a flat file store. So there is no folder hierarchy as in a traditional hard drive. However, S3 allows users to create pseudo-folders by prepending object keys with \code{foldername/}. The \code{put_folder} function is provided as a high-level convenience function for creating folders. This is not actually necessary as objects with slashes in their key will be displayed in the S3 web console as if they were in folders, but it may be useful for creating an empty directory (which is possible in the web console).
}
\examples{
\dontrun{
  library("datasets")
  
  # write file to S3
  tmp <- tempfile()
  on.exit(unlink(tmp))
  utils::write.csv(mtcars, file = tmp)
  put_object(tmp, object = "mtcars.csv", bucket = "myexamplebucket")

  # create a "folder" in a bucket
  put_folder("example", bucket = "myexamplebucket")
  ## write object to the "folder"
  put_object(tmp, object = "example/mtcars.csv", bucket = "myexamplebucket")

  # write serialized, in-memory object to S3
  x <- rawConnection(raw(0), "w")
  utils::write.csv(mtcars, x)
  put_object(rawConnectionValue(x), object = "mtcars.csv", bucket = "myexamplebucketname")

  # use `headers` for server-side encryption
  ## require appropriate bucket policy
  ## encryption can also be set at the bucket-level using \\code{\\link{put_encryption}}
  put_object(file = tmp, object = "mtcars.csv", bucket = "myexamplebucket",
             headers = c('x-amz-server-side-encryption' = 'AES256'))

  # alternative "S3 URI" syntax:
  put_object(rawConnectionValue(x), object = "s3://myexamplebucketname/mtcars.csv")
  close(x)

  # read the object back from S3
  read.csv(text = rawToChar(get_object(object = "s3://myexamplebucketname/mtcars.csv")))
}
}
\references{
\href{http://docs.aws.amazon.com/AmazonS3/latest/API/RESTObjectPUT.html}{API Documentation}
}
\seealso{
\code{\link{put_bucket}}, \code{\link{get_object}}, \code{\link{delete_object}}, \code{\link{put_encryption}}
}
