% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hdi.R
\name{hdi}
\alias{hdi}
\alias{hdi.numeric}
\alias{hdi.stanreg}
\alias{hdi.brmsfit}
\title{Highest Density Interval (HDI)}
\usage{
hdi(x, ...)

\method{hdi}{numeric}(x, ci = 0.9, verbose = TRUE, ...)

\method{hdi}{stanreg}(x, ci = 0.9, effects = c("fixed", "random",
  "all"), parameters = NULL, verbose = TRUE, ...)

\method{hdi}{brmsfit}(x, ci = 0.9, effects = c("fixed", "random",
  "all"), component = c("conditional", "zi", "zero_inflated", "all"),
  parameters = NULL, verbose = TRUE, ...)
}
\arguments{
\item{x}{Vector representing a posterior distribution. Can also be a
\code{stanreg} or \code{brmsfit} model.}

\item{...}{Currently not used.}

\item{ci}{Value or vector of probability of the interval (between 0 and 1)
to be estimated. Named Credible Interval (CI) for consistency.}

\item{verbose}{Toggle off warnings.}

\item{effects}{Should results for fixed effects, random effects or both be returned?
Only applies to mixed models. May be abbreviated.}

\item{parameters}{Regular expression pattern that describes the parameters that
should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}

\item{component}{Should results for all parameters, parameters for the conditional model
or the zero-inflated part of the model be returned? May be abbreviated. Only
applies to \pkg{brms}-models.}
}
\value{
A data frame with following columns:
  \itemize{
    \item \code{Parameter} The model parameter(s), if \code{x} is a model-object. If \code{x} is a vector, this column is missing.
    \item \code{CI} The probability of the HDI.
    \item \code{CI_low} , \code{CI_high} The lower and upper HDI limits for the parameters.
  }
}
\description{
Compute the \strong{Highest Density Interval (HDI)} of a posterior distribution, i.e., all points within the interval have a higher probability density than points outside the interval. The HDI can be used in the context of Bayesian posterior characterisation as \strong{Credible Interval (CI)}.
}
\details{
Unlike equal-tailed intervals (see \link{ci}) that typically exclude 2.5\% from each tail
  of the distribution, the HDI is \emph{not} equal-tailed and therefore always
  includes the mode(s) of posterior distributions.
  \cr \cr
  By default, \code{hdi()} returns the 90\% intervals (\code{ci = 0.9}),
  deemed to be more stable than, for instance, 95\% intervals (\cite{Kruschke, 2015}).
  An effective sample size of at least 10.000 is recommended if 95\% intervals
  should be computed (\cite{Kruschke, 2015, p. 183ff}).
}
\examples{
library(bayestestR)

posterior <- rnorm(1000)
hdi(posterior, ci = .90)
hdi(posterior, ci = c(.80, .90, .95))
\dontrun{
library(rstanarm)
model <- rstanarm::stan_glm(mpg ~ wt + cyl, data = mtcars)
hdi(model)
hdi(model, ci = c(.80, .90, .95))

library(brms)
model <- brms::brm(mpg ~ wt + cyl, data = mtcars)
hdi(model)
hdi(model, ci = c(.80, .90, .95))
}

}
\references{
Kruschke, J. (2015). Doing Bayesian data analysis: A tutorial with R, JAGS, and Stan. Academic Press.
}
\author{
Credits go to \href{https://rdrr.io/cran/ggdistribute/src/R/stats.R}{ggdistribute} and \href{https://github.com/mikemeredith/HDInterval}{HDInterval}.
}
