% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_density.R
\name{estimate_density}
\alias{estimate_density}
\title{Density Estimation}
\usage{
estimate_density(x, method = "kernel", precision = 2^10,
  extend = FALSE, extend_scale = 0.1, bw = "SJ", ...)
}
\arguments{
\item{x}{Vector representing a posterior distribution. Can also be a
\code{stanreg}, \code{brmsfit} or a \code{BayesFactor} model.}

\item{method}{Density estimation method. Can be \code{"kernel"} (default), \code{"logspline"} or \code{"KernSmooth"}.}

\item{precision}{Number of points of density data. See the \code{n} parameter in \link[=density]{density}.}

\item{extend}{Extend the range of the x axis by a factor of \code{extend_scale}.}

\item{extend_scale}{Ratio of range by which to extend the x axis. A value of \code{0.1} means that the x axis will be extended by \code{1/10} of the range of the data.}

\item{bw}{the smoothing bandwidth to be used.  The kernels are scaled
    such that this is the standard deviation of the smoothing kernel.
    (Note this differs from the reference books cited below, and from S-PLUS.)

    \code{bw} can also be a character string giving a rule to choose the
    bandwidth.  See \code{\link{bw.nrd}}. \cr The default,
    \code{"nrd0"}, has remained the default for historical and
    compatibility reasons, rather than as a general recommendation,
    where e.g., \code{"SJ"} would rather fit, see also Venables and
    Ripley (2002).

    The specified (or computed) value of \code{bw} is multiplied by
    \code{adjust}.
  }

\item{...}{Currently not used.}
}
\description{
This function is a wrapper over different methods of density estimation. By default, it uses the base R \link{density} with by default uses a different smoothing bandwidth (\code{"SJ"}) from the legacy default implemented the base R \link{density} function (\code{"nrd0"}). However, Deng \& Wickham suggest that \code{method = "KernSmooth"} is the fastest and the most accurate.
}
\examples{
library(bayestestR)

set.seed(1)
x <- rnorm(250, 1)

# Methods
density_kernel <- estimate_density(x, method = "kernel")
density_logspline <- estimate_density(x, method = "logspline")
density_KernSmooth <- estimate_density(x, method = "KernSmooth")
density_mixture <- estimate_density(x, method = "mixture")

hist(x, prob = TRUE)
lines(density_kernel$x, density_kernel$y, col = "black", lwd = 2)
lines(density_logspline$x, density_logspline$y, col = "red", lwd = 2)
lines(density_KernSmooth$x, density_KernSmooth$y, col = "blue", lwd = 2)
lines(density_mixture$x, density_mixture$y, col = "green", lwd = 2)

# Extension
density_extended <- estimate_density(x, extend = TRUE)
density_default <- estimate_density(x, extend = FALSE)

hist(x, prob = TRUE)
lines(density_extended$x, density_extended$y, col = "red", lwd = 3)
lines(density_default$x, density_default$y, col = "black", lwd = 3)

df <- data.frame(replicate(4, rnorm(100)))
head(estimate_density(df))

# rstanarm models
# -----------------------------------------------
library(rstanarm)
model <- stan_glm(mpg ~ wt + gear, data = mtcars, chains = 2, iter = 200, refresh = 0)
head(estimate_density(model))

library(emmeans)
head(estimate_density(emtrends(model, ~1, "wt")))
\dontrun{
# brms models
# -----------------------------------------------
library(brms)
model <- brms::brm(mpg ~ wt + cyl, data = mtcars)
estimate_density(model)
}

}
\references{
Deng, H., & Wickham, H. (2011). Density estimation in R. Electronic publication.
}
