\name{mle}
\alias{mle2}
\alias{mle}
\alias{calc_mle2_function}
\title{Maximum Likelihood Estimation}
\description{
  Estimate parameters by the method of maximum likelihood.
}
\usage{
mle2(minuslogl, start, method, optimizer,
    fixed = NULL, data=NULL,
    subset=NULL,
default.start=TRUE, eval.only = FALSE, vecpar=FALSE,
parameters=NULL,
skip.hessian=FALSE,trace=FALSE,\dots)
calc_mle2_function(formula,parameters,start,data=NULL,trace=FALSE)
}
\arguments{
  \item{minuslogl}{Function to calculate negative log-likelihood.}
  \item{start}{Named list. Initial values for optimizer.}
  \item{method}{Optimization method to use. See \code{\link{optim}}.}
  \item{optimizer}{Optimization function to use. (Stub.)}
  \item{fixed}{Named list.  Parameter values to keep fixed during
    optimization.}
  \item{data}{list of data to pass to minuslogl}
  \item{subset}{logical vector for subsetting data (STUB)}
  \item{default.start}{Logical: allow default values of \code{minuslogl}
    as starting values?}
  \item{eval.only}{Logical: return value of \code{minuslogl(start)}
    rather than optimizing}
  \item{vecpar}{Logical: is first argument a vector of all parameters?
    (For compatibility with \code{\link{optim}}.)}
  \item{parameters}{List of linear models for parameters}
  \item{\dots}{Further arguments to pass to optimizer}
  \item{formula}{a formula for the likelihood (see Details)}
  \item{trace}{Logical: print parameter values tested?}
  \item{skip.hessian}{Bypass Hessian calculation?}
}
\details{
  The \code{\link{optim}} optimizer is used to find the minimum of the
  negative log-likelihood.  An approximate covariance matrix for the
  parameters is obtained by inverting the Hessian matrix at the optimum.

  The \code{minuslogl} argument can also specify a formula,
  rather than an objective function, of the
  form \code{x~ddistn(param1,...,paramn)}.  In this case
  \code{ddistn} is taken to be a probability or density
  function, which must have (literally) \code{x} as its
  first argument (although this argument may be interpreted as
  a matrix of multivariate responses) and which must have
  a \code{log} argument that can be used to specify the
  log-probability or log-probability-density is required.
  If a formula is specified, then \code{parameters} can contain
  a list of linear models for the parameters.

  If a formula is given and non-trivial linear models are given
  in \code{parameters} for some of the variables, then
  model matrices will be generated using \code{model.matrix}:
  \code{start} can either be an exhaustive list of starting values (in
  the order given by \code{model.matrix}) or values can be given just
  for the higher-level parameters: in this case, all of the additional
  parameters generated by \code{model.matrix} will be given starting
  values of zero.

  The \code{trace} argument applies only when a formula is specified.
  If you specify a function, you can build in your own \code{print()}
  or \code{cat()} statement to trace its progress.  (You can also
  specify a value for \code{trace} as part of a \code{control}
  list for \code{optim()}: see \code{\link{optim}}.)

  The \code{skip.hessian} argument is useful if the function is
  crashing with a "non-finite finite difference value" error when trying
  to evaluate the Hessian, but will preclude many subsequent
  confidence interval calculations.  (You will know the Hessian
  is failing if you use \code{method="Nelder-Mead"} and still
  get a finite-difference error.)
  
  If convergence fails, see \code{\link{optim}}
  for the meanings of the error codes.
}
\value{
  An object of class \code{"mle2"}.
}
\note{
  Note that the \code{minuslogl} function should
  return -log L (not -2 log L). It
  is the user's responsibility
  to ensure that the likelihood is correct, and that
  asymptotic likelihood inference is valid (e.g.
  that there are "enough" data and that the
  estimated parameter values do not lie on the
  boundary of the feasible parameter space).
}
\seealso{
  \code{\link{mle2-class}}
}
\examples{
x <- 0:10
y <- c(26, 17, 13, 12, 20, 5, 9, 8, 5, 4, 8)
LL <- function(ymax=15, xhalf=6)
    -sum(stats::dpois(y, lambda=ymax/(1+x/xhalf), log=TRUE))
## uses default parameters of LL
(fit <- mle2(LL))
mle2(LL, fixed=list(xhalf=6))

(fit0 <- mle2(y~dpois(lambda=ymean),start=list(ymean=mean(y))))
anova(fit0,fit)
summary(fit)
logLik(fit)
vcov(fit)
p1 <- profile(fit)
plot(p1, absVal=FALSE)
confint(fit)

## use bounded optimization
## the lower bounds are really > 0, but we use >=0 to stress-test profiling
(fit1 <- mle2(LL, method="L-BFGS-B", lower=c(0, 0)))
p1 <- profile(fit1)
plot(p1, absVal=FALSE)

## a better parameterization:
LL2 <- function(lymax=log(15), lxhalf=log(6))
    -sum(stats::dpois(y, lambda=exp(lymax)/(1+x/exp(lxhalf)), log=TRUE))
(fit2 <- mle2(LL2))
plot(profile(fit2), absVal=FALSE)
exp(confint(fit2))
vcov(fit2)
cov2cor(vcov(fit2))

mle2(y~dpois(lambda=exp(lymax)/(1+x/exp(lhalf))),
   start=list(lymax=0,lhalf=0),
   parameters=list(lymax~1,lhalf~1))


## try bounded optimization with nlminb and constrOptim
(fit1B <- mle2(LL, optimizer="nlminb", lower=c(1e-7, 1e-7)))
p1B <- profile(fit1B)
confint(p1B)
(fit1C <- mle2(LL, optimizer="constrOptim", ui = c(1,1), ci=2,
   method="Nelder-Mead"))
}

\keyword{models}

