#' Estimate Multiple Correlation Coefficients
#' (R-Squared and Adjusted R-Squared)
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details R-squared (\eqn{R^{2}}) and
#'   adjusted R-squared (\eqn{\bar{R}^{2}})
#'   are derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{R^{2}} and \eqn{\bar{R}^{2}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{object}{The function argument `object`.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{R^{2}} and \eqn{\bar{R}^{2}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{R^{2}} and \eqn{\bar{R}^{2}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{R^{2}} and \eqn{\bar{R}^{2}}.}
#'     \item{fun}{Function used ("RSqMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#' @examples
#' # Fit the regression model
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#' # Generate the sampling distribution of parameter estimates
#' # (use a large R, for example, R = 20000 for actual research)
#' mc <- MC(object, R = 100)
#' # Generate confidence intervals for standardized regression slopes
#' rsq <- RSqMC(mc)
#' # Methods --------------------------------------------------------
#' print(rsq)
#' summary(rsq)
#' coef(rsq)
#' vcov(rsq)
#' confint(rsq, level = 0.95)
#' @export
#' @family Beta Monte Carlo Functions
#' @keywords betaMC rsq
RSqMC <- function(object) {
  stopifnot(
    methods::is(
      object,
      "mc"
    )
  )
  thetahatstar <- lapply(
    X = object$thetahatstar,
    FUN = function(x) {
      rsq <- .RSqofSigma(
        sigmacap = x$sigmacap,
        k = object$lm_process$k
      )
      adj <- (
        1 - (
          1 - rsq
        ) * (
          (
            object$lm_process$n - 1
          ) / object$lm_process$df
        )
      )
      return(c(rsq = rsq, adj = adj))
    }
  )
  out <- list(
    call = match.call(),
    object = object,
    thetahatstar = thetahatstar,
    est = object$lm_process$rsq,
    fun = "RSqMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  return(
    out
  )
}
