% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_dynamic_growth.R
\name{fit_dynamic_growth}
\alias{fit_dynamic_growth}
\title{Fit dynamic growth models}
\usage{
fit_dynamic_growth(
  fit_data,
  env_conditions,
  starting_point,
  known_pars,
  sec_model_names,
  ...,
  check = TRUE,
  formula = logN ~ time
)
}
\arguments{
\item{fit_data}{Tibble with the data to use for model fit. It must
contain a column with the elapsed time (named "time" by default) and another one
with the decimal logarithm of the observed population size (named "logN" by default).
Different column names can be specified using the "formula" argument.}

\item{env_conditions}{Tibble with the (dynamic) environmental conditions
during the experiment. It must have one column with the
elapsed time (named "time" by default) and as many columns as required with the 
environmental conditions. A different column name can be specified using the 
"formula" argument, although it must be the same one as in "fit_data".
Note that only those defined in "sec_model_names" will be considered for
the model fit.}

\item{starting_point}{A named vector of starting values for the model parameters.
Parameters for the primary model must be named in the usual way. Parameters for the
secondary model are named as \code{env_factor}+'_'+\code{parameter}. For instance,
the maximum growth temperature shall be named 'temperature_xmax'.}

\item{known_pars}{A named vectors of known model parameters (i.e. not fitted). They
must be named using the same convention as for \code{starting_point}.}

\item{sec_model_names}{A named character vector defining the secondary model for each
environmental factor. The names define the factor and the value the type of model.
Names must match columns in \code{fit_data} and \code{env_conditions}.}

\item{...}{Additional arguments passed to modFit.}

\item{check}{Whether to check model parameters (TRUE by default).}

\item{formula}{an object of class "formula" describing the x and y variables.
\code{logN ~ time} as a default.}
}
\value{
An instance of \code{\link{FitDynamicGrowth}}.
}
\description{
Fits a growth model to a data obtained under dynamic conditions
using the one-step approach (non-linear regression).
}
\examples{
\donttest{
## We use the datasets included in the package

data("example_dynamic_growth")
data("example_env_conditions")

## Define the secondary models

sec_model_names <- c(temperature = "CPM", aw= "CPM")

## Any model parameter can be fixed

known_pars <- list(Nmax = 1e4,  # Primary model
    N0 = 1e0, Q0 = 1e-3,  # Initial values of the primary model
    mu_opt = 4, # mu_opt of the gamma model
    temperature_n = 1,  # Secondary model for temperature
    aw_xmax = 1, aw_xmin = .9, aw_n = 1  # Secondary model for water activity
    )

## The remaining parameters need initial values

my_start <- list(temperature_xmin = 25, temperature_xopt = 35,
    temperature_xmax = 40, aw_xopt = .95)

## We can now call the fitting function

my_dyna_fit <- fit_dynamic_growth(example_dynamic_growth, example_env_conditions,
    my_start, known_pars, sec_model_names)

summary(my_dyna_fit)

## We can compare the data and the fitted curve

plot(my_dyna_fit)

## We can plot any environmental condition using add_factor

plot(my_dyna_fit, add_factor = "aw",
    label_y1 = "Log count (log CFU/ml)",
    label_y2 = "Water activity")
}

}
