% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wBuild.R
\name{wBuild}
\alias{wBuild}
\title{Derive parameters for building integration grids}
\usage{
wBuild(f, init, dim.theta2 = length(init), approx = "gaussian",
  link = rep("identity", length(init)), link.params = rep(list(NA),
  length(init)), optim.control = list(maxit = 5000, method = "BFGS"),
  ...)
}
\arguments{
\item{f}{function used to derive the weight function \eqn{w}.
\code{f} must be able to be called via \code{f(par, log, ...)}}

\item{init}{initial guess for mode of \code{f}.}

\item{dim.theta2}{\code{wBuild} assumes \code{par} is partitioned such that 
\code{par=c(theta1,theta2)}.  \code{dim.theta2} specifies the size of the 
partition.  The default is to assume that \code{f} is defined without a 
\code{theta1} component.}

\item{approx}{Style of approximation (i.e., \eqn{w}) to be created from mode 
of \code{f}.
\describe{
  \item{\code{'gaussian'}}{Gaussian approximation for \code{theta2} at 
    the mode of \code{f}. Assumes \code{f} is proportional to the marginal 
    posterior density for \code{theta2}.}
  \item{\code{'condgauss'}}{Gaussian approximation for \code{theta2} at 
    the mode of \code{f}.  The approximation is conditioned on the value of
    the mode for \code{theta1}. Assumes \code{f} is proportional to the 
    joint posterior density for \code{theta1,theta2.}}
  \item{\code{'condgauss-laplace'}}{Gaussian approximation for 
    \code{theta2} at the mode of \code{f}.  The approximation is 
    conditioned on a separate laplace approximation of the marginal 
    posterior mode for \code{theta1}.  Assumes \code{f} is proportional to 
    the joint posterior density for \code{theta1,theta2.}}
  \item{\code{'margauss'}}{Gaussian approximation for 
    \code{theta2} at the mode of \code{f}.  Assumes \code{f} is 
    proportional to the joint posterior density for \code{theta1,theta2.}, 
    then uses the marginal mean and covariance from the posterior's 
    gaussian approximation.}
}}

\item{link}{character vector that specifies transformations used during
optimization and integration of \eqn{f(\theta_2 | X)}.  While
\eqn{\theta_2} may be defined on arbitrary support, \code{wtdMix} performs
optimization and integration of \eqn{\theta_2} on an unconstrained support.
The \code{link} vector describes the transformations that must be applied
to each element of \eqn{\theta_2}.  Jacobian functions for the
transformations will automatically be added to the optimization and
integration routines. Currently supported link functions are \code{'log'},
\code{'logit'}, and \code{'identity'}.}

\item{link.params}{Optional list of additional parameters for link
functions.  For example, the logit function can be extended to allow
mappings to any closed interval.   There should be one list entry for each
link function.  Specify NA if no additional arguments are passed.}

\item{optim.control}{List of arguments to pass to \code{stat::optim}
  when used to find mode of \code{f}.
\describe{
  \item{\code{maxit}}{Maximum number of iterations to run \code{optim} 
    for.}
  \item{\code{method}}{Optimization routine to use with \code{optim}.}
}}

\item{...}{additional arguments needed for function evaluation.}
}
\description{
Note: \eqn{w} is defined on the transformed scale, but for convenience 
\code{f} is defined on the original scale.
}
\examples{
# Use BISQuE to approximate the marginal posterior distribution for unknown
# population f(N|c, r) for the fur seals capture-recapture data example in 
# Givens and Hoeting (2013), example 7.10.

data('furseals')

# define theta transformation and jacobian
tx.theta = function(theta) { 
  c(log(theta[1]/theta[2]), log(sum(theta[1:2]))) 
}
itx.theta = function(u) { 
  c(exp(sum(u[1:2])), exp(u[2])) / (1 + exp(u[1])) 
}
lJ.tx.theta = function(u) {
  log(exp(u[1] + 2*u[2]) + exp(2*sum(u[1:2]))) - 3 * log(1 + exp(u[1]))
}

# compute constants
r = sum(furseals$m)
nC = nrow(furseals)

# set basic initialization for parameters
init = list(U = c(-.7, 5.5))
init = c(init, list(
  alpha = rep(.5, nC),
  theta = itx.theta(init$U),
  N = r + 1
))


post.alpha_theta = function(theta2, log = TRUE, ...) {
  # Function proportional to f(alpha, U1, U2 | c, r) 
  
  alpha = theta2[1:nC]
  u = theta2[-(1:nC)]
  theta = itx.theta(u)
  p = 1 - prod(1-alpha)
  
  res = - sum(theta)/1e3 - r * log(p) + lJ.tx.theta(u) - 
    nC * lbeta(theta[1], theta[2])
  for(i in 1:nC) {
    res = res + (theta[1] + furseals$c[i] - 1)*log(alpha[i]) + 
      (theta[2] + r - furseals$c[i] - 1)*log(1-alpha[i])
  }
  
  if(log) { res } else { exp(res) }
}

post.N.mixtures = function(N, params, log = TRUE, ...) {
  # The mixture component of the weighted mixtures for f(N | c, r)
  dnbinom(x = N-r, size = r, prob = params, log = log)
}

mixparams.N = function(theta2, ...) {
  # compute parameters for post.N.mixtures
  1 - prod(1 - theta2[1:nC])
}


w.N = wBuild(f = post.alpha_theta, init = c(init$alpha, init$U), 
             approx = 'gauss', link = c(rep('logit', nC), rep('identity', 2)))

m.N = wMix(f1 = post.N.mixtures, f1.precompute = mixparams.N, 
           f2 = post.alpha_theta, w = w.N)



# compute posterior mean
m.N$expectation$Eh.precompute(h = function(p) ((1-p)*r/p + r), 
                                   quadError = TRUE)

# compute posterior density
post.N.dens = data.frame(N = r:105)
post.N.dens$d = m.N$f(post.N.dens$N)

# plot posterior density
plot(d~N, post.N.dens, ylab = expression(f(N~'|'~bold(c),r)))

}
