% Generated by roxygen2 (4.0.2): do not edit by hand
\name{blocks}
\alias{blocks}
\title{Block designs}
\usage{
blocks(treatments, replicates, blocklevels = NULL, searches = NULL,
  seed = NULL)
}
\arguments{
\item{treatments}{a list giving a partition of the total number of treatments into equally replicated treatment sets.}

\item{replicates}{a list assigning a replication level to each set in the \code{treatments} list.}

\item{blocklevels}{an optional list of levels where the first level is the number of main blocks and the remaining
levels, if any, are the levels of a hierarchy of nested sub-blocks in descending order of nesting.
The default is an orthogonal main blocks design.}

\item{searches}{an optional integer for the number of local optima searched during an optimization.
The default is the minimum of 32 or the ceiling of 4096 divided by the number of units.}

\item{seed}{an optional integer seed for initializing the random number generator. The default
is a random seed.}
}
\value{
\item{Design}{Data frame showing the listing of treatments allocated to blocks}
\item{Plan}{Data frame showing a plan of treatments allocated to sub-plots within blocks}
\item{Incidences}{List of blocks-by-treatments incidence matrices, one for each stratum of the design}
\item{Efficiencies}{Data frame showing the A-efficiency factor for each stratum of the design together with an upper bound, where available}
\item{seed}{Numerical seed for random number generator}
}
\description{
Constructs nested block designs for unstructured treatment sets with arbitrary replication,
not necessarily all equal, and arbitrary depth of nesting.
}
\details{
\code{blocks(...)} constructs general block designs by using a swapping algorithm to
maximize the determinant of
the information matrix (D-optimality). The algorithm proceeds by making improving swaps between blocks
 in the top stratum
of a design until no further improvement is possible and then repeating
the process for each nested stratum in turn until the bottom stratum is reached.
At each stage, improving swaps are made within the blocks of any existing strata
to ensure top-down optimization. Certain special lattice block designs with v**2 equally
replicated treatments in blocks of size v and with k replicates
 where k <= 3 for any v, or k <= v+1 for prime or prime-power v, or k <= 4 for v = 10 have algebraic optima and
 these designs are constructed algebraically.

The treatments design is defined by the \code{treatments} and the \code{replicates} parameter lists which
partition the total required number of treatments into
sets of equally replicated treatments. The \code{treatments} list defines the size of each set
and the \code{replicates} list defines the replication. The two lists must be of equal length and the sets
must be in the same order in both lists. Treatments are numbered consecutively according to the set order
and treatments with the same replication can be split between two or more sets if a non-standard treatment order
is required.

The blocks design is defined by the \code{blocklevels} list which contains the nested blocks levels
for each stratum of the design. The first level is the number of main blocks
and the successive levels, if any, are the numbers of nested sub-blocks in each stratum of
 a hierarchy of nested sub-blocks.
The length of the list is the number of strata in the design and the
running products of the levels are the total blocks in each successive strata of the
design. The blocks in any given stratum are always equal in size or differ by, at most, a
single unit. The default is the highest common factor of the replication levels,
which gives a main blocks design with a maximal set of complete orthogonal main blocks.

The \code{searches} parameter is the number of local optima searched during an optimization.
Increasing the number of searches may improve the efficiency of a design but
 will also increase the search time.

The \code{seed} parameter is an integer used to initialize the random number generator. The
 default is a random integer but any fixed positive integer can be used instead,if required.

Blocks and treatments are fully randomized within the constraints of a nested blocks design.
}
\examples{
# 3 treatments with 2 reps, 2 treatments with 4 reps, 4 treatments with 3 reps
# the hcf of the replication numbers is 1 and the default design is a completely randomized design
blocks(treatments=c(3,2,4),replicates=c(2,4,3))

# 50 treatments with 4 reps in 4 complete randomized blocks
blocks(treatments=50,replicates=4)

# as above but with 4 main blocks and 5 nested blocks within each main block
blocks(treatments=50,replicates=4,blocklevels=c(4,5))

# as above but with 20 additional single replicate treatments, one to each block
blocks(treatments=c(50,20),replicates=c(4,1),blocklevels=c(4,5))

# 64 treatments with 2 reps and 2 main blocks with five 2-level nested factors
blocks(treatments=64,replicates=2,blocklevels=c(2,2,2,2,2,2),searches=4)

# concurrence matrices of 36 treatments with 3 reps and 3 main blocks with 6 nested blocks
crossprod(blocks(treatments=36,replicates=3,blocklevels=c(3,6))$Incidences[[2]])

# concurrence matrix for 13 treatments with 4 reps and 13 treatments with one rep in 13 blocks
crossprod(blocks(c(13,13),c(4,1),13,searches=100)$Incidences[[1]])
}
\references{
Sailer, M. O. (2013). crossdes: Construction of Crossover Designs. R package version 1.1-1. http://CRAN.R-project.org/package=crossdes
}

