% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/r2dtype.R
\name{r_to_dtype}
\alias{r_to_dtype}
\alias{dtype_to_r}
\title{Convert from or to ZARR data types}
\usage{
r_to_dtype(x, dtype, na_value = NA, ...)

dtype_to_r(x, dtype, na_value = NA, ...)
}
\arguments{
\item{x}{Object to be converted}

\item{dtype}{The data type used for encoding/decoding raw data. The \code{dtype} is
a code consisting of at least 3 characters. The first character indicates the
\href{https://en.wikipedia.org/wiki/Endianness}{endianness} of the data: \code{'<'}
(little-endian), \code{'>'} (big-endian), or \code{'|'} (endianness not relevant).

The second character represents the main data type (\code{'b'} boolean (logical),
\code{'i'} signed integer, \code{'u'} unsigned integer, \code{'f'} floating point number,
\code{'c'} complex number). \code{'M'} is used for date-time objects and \code{'m'} for delta
time (see \code{\link[=difftime]{difftime()}}). \code{'S'} for UTF8 encoded character strings and \code{'U'} for
UTF32 encoded character strings.

The following characters are numerical indicating the byte size of the data type.
For example: \code{dtype = "<f4"} means a 32 bit floating point number; \code{dtype = "|b1"}
means an 8 bit logical value. An exception is the main type \code{'U'}, where the number
indicates the number of characters, where each character is represented by 4 bytes.

The main types \code{'M'} and \code{'m'} should always be ended with the time unit between
square brackets for storing the date time (difference). A valid code would be \verb{"<M8[h]}.

For more details about dtypes see
\href{https://zarr-specs.readthedocs.io/en/latest/v2/v2.0.html}{ZARR V2.0}
or \code{vignette("dtypes")}.}

\item{na_value}{When storing raw data, you may want to reserve a value to
represent missing values. This is also what \code{R} does for \code{NA} values.
Other software may use different values to represent missing values. Also,
some data types have insufficient storage capacity to store \code{R} \code{NA} values.

Therefore, you can use this argument to indicate which value should represent
missing values. By default it uses \code{R} \code{NA}. When set to \code{NULL}, missing values
are just processed as is, without any further notice or warning.

For more details see \code{vignette("dtypes")}.}

\item{...}{Ignored}
}
\value{
In case of \code{r_to_dtype()} a vector of encoded \code{raw} data is returned.
In case of \code{dtype_to_r()} a vector of an R type (appropriate for the specified \code{dtype})
is returned if possible.
}
\description{
Use \href{https://zarr-specs.readthedocs.io/en/latest/v2/v2.0.html}{ZARR V2.0} data types to
convert between R native types and raw data.
}
\details{
One of the applications of BLOSC compression is in ZARR, which is used to store
n-dimensional structured data. \code{r_to_dtype()} and \code{dtype_to_r()} are convenience functions
that allows you to convert most common data types to R native types.

R natively only supports \code{\link[=logical]{logical()}} (actually stored as 32 bit integer in memory),
\code{\link[=integer]{integer()}} (signed 32 bit integers), \code{\link[=numeric]{numeric()}} (64 bit floating points) and \code{\link[=complex]{complex()}}
(real and imaginary component both represented by a 64 bit floating point). R also has some
more complex classes, but those are generally derivatives of the aforementioned types.

The functions documented here will attempt to convert raw data to R types (or vice versa).
As not all 'dtypes' have an appropriate R type counterpart, some conversions will not
be possible directly and will result in an error.

For more details see \code{vignette("dtypes")}.
}
\examples{
## Encode volcano data to 16 bit floating point values
volcano_encoded <-
  r_to_dtype(volcano, dtype = "<f2")

## Decode the volcano format to its original
volcano_reconstructed <-
  dtype_to_r(volcano_encoded, dtype = "<f2")

## The reconstruction is the same as its original:
all(volcano_reconstructed == volcano)

## Encode a numeric sequence with a missing value represented by -999
r_to_dtype(c(1, 2, 3, NA, 4), dtype = "<i2", na_value = -999)
}
\author{
Pepijn de Vries
}
