linear_predictor <- function(draws, i = NULL) {
  # compute the linear predictor (eta) for brms models
  # Args:
  #   draws: a list generated by extract_draws containing
  #          all required data and posterior samples
  #   i: An optional vector indicating the observation(s) 
  #      for which to compute eta. If NULL, eta is computed 
  #      for all all observations at once.
  # Returns:
  #   Usually an S x N matrix where S is the number of samples
  #   and N is the number of observations or length of i if specified. 
  if (length(i) == 1L && is_categorical(draws$f) && 
      isTRUE(draws$old_cat == 2L)) {
    # for some time categorical models were using mv syntax
    nobs <- draws$data$N_trait * (draws$data$ncat - 1)
    i <- seq(i, nobs, draws$data$N_trait)
  }
  N <- ifelse(!is.null(i), length(i), draws$data$N) 
  
  eta <- matrix(0, nrow = draws$nsamples, ncol = N)
  if (!is.null(draws[["b"]])) {
    eta_fe <- fe_predictor(X = p(draws$data[["X"]], i), b = draws[["b"]])
    eta <- eta + eta_fe
  }
  if (!is.null(draws$data$offset)) {
    eta_offset <- rep(p(draws$data$offset, i), draws$nsamples)
    eta <- eta + matrix(eta_offset, ncol = N, byrow = TRUE)
  }
  # incorporate monotonic effects
  monef <- names(draws[["bmo"]])
  for (j in seq_along(monef)) {
    # prepare monotonic group-level effects
    rmo_temp <- draws[["rmo"]][[monef[j]]]
    rmo <- named_list(names(rmo_temp))
    for (g in names(rm)) {
      rmo[[g]] <- re_predictor(
        Z = p(draws[["Zmo"]][[g]], i), r = rmo[[g]]
      )
    }
    eta <- eta + 
      mo_predictor(
        X = p(draws$data$Xmo[, j], i), 
        b = draws[["bmo"]][[j]], 
        simplex = draws$simplex[[j]],
        r = Reduce("+", rmo)
      )
  }
  # incorporate noise-free effects
  meef <- names(draws[["bme"]])
  if (length(meef)) {
    eval_list <- list()
    for (j in seq_along(draws[["Xme"]])) {
      eval_list[[paste0("Xme_", j)]] <- 
        p(draws[["Xme"]][[j]], i, row = FALSE)
    }
    for (j in seq_along(draws[["Cme"]])) {
      eval_list[[paste0("Cme_", j)]] <- 
        p(draws[["Cme"]][[j]], i, row = FALSE)
    }
    calls <- attr(draws[["bme"]], "calls")
    for (j in seq_along(meef)) {
      # prepare noise-free group-level effects
      rme_temp <- draws[["rme"]][[meef[j]]]
      rme <- named_list(names(rme_temp))
      for (g in names(rme)) {
        rme[[g]] <- re_predictor(
          Z = p(draws[["Zme"]][[g]], i), r = rme_temp[[g]]
        )
      }
      eta <- eta + 
        me_predictor(
          eval_list, call = calls[[j]],
          b = draws[["bme"]][[j]],
          r = Reduce("+", rme)
        )
    }
  }
  # incorporate group-level effects
  group <- names(draws[["r"]])
  for (g in group) {
    eta_re <- re_predictor(
      Z = p(draws[["Z"]][[g]], i), 
      r = draws[["r"]][[g]]
    )
    eta <- eta + eta_re
  }
  # incorporate smooths
  smooths <- names(draws[["s"]])
  for (k in seq_along(smooths)) {
    nb <- seq_len(length(draws[["s"]][[smooths[k]]]))
    for (j in nb) {
      Zs <- p(draws[["Zs"]][[smooths[k]]][[j]], i)
      s <- draws[["s"]][[smooths[k]]][[j]]
      eta <- eta + fe_predictor(X = Zs, b = s)
    }
  }
  if (!is.null(draws[["arr"]])) {
    eta <- eta + fe_predictor(X = p(draws$data$Yarr, i), b = draws[["arr"]])
  }
  if (any(c("ar", "ma") %in% names(draws))) {
    if (!is.null(i)) {
      stop2("Pointwise evaluation is not yet implemented for ARMA models.")
    }
    eta <- arma_predictor(
      standata = draws$data, ar = draws[["ar"]], 
      ma = draws[["ma"]], eta = eta, link = draws$f$link
    )
  }
  if (!is.null(draws$loclev)) {
    eta <- eta + p(draws$loclev, i, row = FALSE)
  }
  if (is_ordinal(draws$f)) {
    if (!is.null(draws[["cs"]]) || !is.null(draws[["rcs"]])) {
      ncat <- draws$data$ncat
      if (!is.null(draws[["rcs"]])) {
        rcs <- named_list(seq_len(ncat - 1))
        for (k in names(rcs)) {
          rcs[[k]] <- named_list(names(draws[["rcs"]][[k]]))
          for (g in names(rcs[[k]])) {
            rcs[[k]][[g]] <- re_predictor(
              Z = p(draws[["Zcs"]][[g]], i),
              r = draws[["rcs"]][[k]][[g]]
            )
          }
          rcs[[k]] <- Reduce("+", rcs[[k]])
        }
      } else {
        rcs <- NULL
      }
      eta <- cs_predictor(
        X = p(draws$data[["Xcs"]], i), 
        b = draws[["cs"]], eta = eta, 
        ncat = ncat, r = rcs
      )
    } else {
      eta <- array(eta, dim = c(dim(eta), draws$data$ncat - 1))
    } 
    for (k in seq_len(draws$data$ncat - 1)) {
      if (draws$f$family %in% c("cumulative", "sratio")) {
        eta[, , k] <- draws$Intercept[, k] - eta[, , k]
      } else {
        eta[, , k] <- eta[, , k] - draws$Intercept[, k]
      }
    }
  } else if (isTRUE(draws$old_cat > 0L)) {
    if (draws$old_cat == 1L) {
      # deprecated as of brms > 0.8.0
      if (!is.null(draws[["cs"]])) {
        eta <- cs_predictor(
          X = p(draws$data[["X"]], i), b = draws[["cs"]], 
          eta = eta, ncat = draws$data$ncat
        )
      } else {
        eta <- array(eta, dim = c(dim(eta), draws$data$ncat - 1))
      }
    } else if (draws$old_cat == 2L) {
      # deprecated as of brms > 0.10.0
      ncat1 <- draws$data$ncat - 1 
      eta <- array(eta, dim = c(nrow(eta), ncol(eta) / ncat1, ncat1))
    }
  }
  unname(eta)
}

linear_predictor_mv <- function(draws, i = NULL) {
  # compute the linear predictor Eta for multivariate models
  # Returns:
  #   An array of dimension Nsamples (or length(i)) x Nobs x Nresp
  resp <- names(draws[["mv"]])
  tmp <- named_list(resp)
  for (r in resp) {
    tmp[[r]] <- linear_predictor(draws[["mv"]][[r]], i = i)
  }
  aperm(do.call(abind::abind, c(tmp, along = 0)), perm = c(2, 3, 1))
}

nonlinear_predictor <- function(draws, i = NULL) {
  # compute the non-linear predictor (eta) for brms models
  # Args:
  #   draws: a list generated by extract_draws containing
  #          all required data and posterior samples
  #   i: An optional vector indicating the observation(s) 
  #      for which to compute eta. If NULL, eta is computed 
  #      for all all observations at once.
  # Returns:
  #   Usually an S x N matrix where S is the number of samples
  #   and N is the number of observations or length of i if specified. 
  nlmodel_list <- list()
  nlpars <- names(draws$nlpars)
  for (nlp in nlpars) {
    nlmodel_list[[nlp]] <- linear_predictor(draws$nlpars[[nlp]], i = i)
  }
  for (cov in names(draws$C)) {
    nlmodel_list[[cov]] <- p(draws$C[[cov]], i, row = FALSE)  
  }
  # evaluate non-linear predictor
  out <- try(with(nlmodel_list, eval(draws$nlform)), silent = TRUE)
  if (is(out, "try-error")) {
    if (grepl("could not find function", out)) {
      out <- rename(out, "Error in eval(expr, envir, enclos) : ", "")
      stop2(out, " Most likely this is because you used a Stan ",
            "function in the non-linear model formula that ",
            "is not defined in R. Currently, you have to write ",
            "this function yourself making sure that it is ",
            "vectorized. I apologize for the inconvenience.")
    } else {
      out <- rename(out, "^Error :", "", fixed = FALSE)
      stop2(out)
    }
  }
  unname(out)
}

fe_predictor <- function(X, b) {
  # compute eta for fixed effects
  # Args:
  #   X: fixed effects design matrix
  #   b: fixed effects samples
  stopifnot(is.matrix(X))
  stopifnot(is.matrix(b))
  tcrossprod(b, X)
}

mo_predictor <- function(X, b, simplex, r = NULL) {
  # compute eta for monotonic effects
  # Args:
  #   X: a vector of data for the monotonic effect
  #   b: monotonic effects samples
  #   simplex: matrix of samples of the simplex
  #            corresponding to bm
  #   r: matrix with monotonic group-level samples
  stopifnot(is.vector(X))
  stopifnot(is.matrix(simplex))
  b <- as.vector(b)
  for (i in 2:ncol(simplex)) {
    # compute the cumulative representation of the simplex 
    simplex[, i] <- simplex[, i] + simplex[, i - 1]
  }
  simplex <- cbind(0, simplex)
  if (is.null(r)) r <- 0 
  (b + r) * simplex[, X + 1]
}

me_predictor <- function(eval_list, call, b, r = NULL) {
  # compute eta for noise-free effects
  # Args:
  #   Xme: a matrix of samples of the noise-free variable
  #   b: samples of the noise-free coefficient
  #   r: matrix with meef group-level samples
  b <- as.vector(b)
  if (is.null(r)) r <- 0 
  (b + r) * eval(call, eval_list)
}

re_predictor <- function(Z, r) {
  # compute eta for random effects
  # Args:
  #   Z: sparse random effects design matrix
  #   r: random effects samples
  # Returns: 
  #   linear predictor for random effects
  Matrix::as.matrix(Matrix::tcrossprod(r, Z))
}

cs_predictor <- function(X, b, eta, ncat, r = NULL) {
  # add category specific effects to eta
  # Args:
  #   X: category specific design matrix 
  #   b: category specific effects samples
  #   ncat: number of categories
  #   eta: linear predictor matrix
  #   r: list of samples of cs group-level effects
  # Returns: 
  #   linear predictor including category specific effects as a 3D array
  stopifnot(is.null(X) && is.null(b) || is.matrix(X) && is.matrix(b))
  ncat <- max(ncat)
  eta <- array(eta, dim = c(dim(eta), ncat - 1))
  if (!is.null(X)) {
    I <- seq(1, (ncat - 1) * ncol(X), ncat - 1) - 1
    X <- t(X)
  }
  for (k in seq_len(ncat - 1)) {
    if (!is.null(X)) {
      eta[, , k] <- eta[, , k] + b[, I + k, drop = FALSE] %*% X 
    }
    if (!is.null(r[[k]])) {
      eta[, , k] <- eta[, , k] + r[[k]]
    }
  }
  eta
}

arma_predictor <- function(standata, eta, ar = NULL, ma = NULL, 
                           link = "identity") {
  # compute eta for ARMA effects
  # ToDo: use C++ for this function
  # Args:
  #   standata: the data initially passed to Stan
  #   eta: previous linear predictor samples
  #   ar: autoregressive samples (can be NULL)
  #   ma: moving average samples (can be NULL)
  #   link: the link function as character string
  # Returns:
  #   new linear predictor samples updated by ARMA effects
  S <- nrow(eta)
  Kar <- ifelse(is.null(ar), 0, ncol(ar))
  Kma <- ifelse(is.null(ma), 0, ncol(ma))
  K <- max(Kar, Kma, 1)
  Ks <- 1:K
  Y <- link(standata$Y, link)
  N <- length(Y)
  tg <- c(rep(0, K), standata$tg)
  E <- array(0, dim = c(S, K, K + 1))
  e <- matrix(0, nrow = S, ncol = K)
  zero_mat <- e
  zero_vec <- rep(0, S)
  for (n in 1:N) {
    if (Kma) {
      # add MA effects
      eta[, n] <- eta[, n] + rowSums(ma * E[, 1:Kma, K])
    }
    e[, K] <- Y[n] - eta[, n]
    if (n < N) {
      I <- which(n < N & tg[n + 1 + K] == tg[n + 1 + K - Ks])
      E[, I, K + 1] <- e[, K + 1 - I]
    }
    if (Kar) {
      # add AR effects
      eta[, n] <- eta[, n] + rowSums(ar * E[, 1:Kar, K])
    }
    # allows to keep the object size of e and E small
    E <- abind(E[, , 2:(K + 1), drop = FALSE], zero_mat)
    if (K > 1) {
      e <- cbind(e[, 2:K, drop = FALSE], zero_vec)
    }
  }
  eta
}

get_eta <- function(draws, i = NULL) {
  # extract the linear predictor of observation i from draws
  # Args:
  #   draws: a list generated by extract_draws
  #   i: either NULL or a vector (typically of length 1) 
  #      indicating the observations for which to extract eta
  # Returns:
  #   An S x N matrix, where N is the number of extracted observations
  if (is.numeric(draws)) {
    if (is.null(dim(draws))) {
      draws <- as.matrix(draws)
    }
    if (is.null(i)) {
      eta <- draws
    } else if (length(dim(draws)) == 3L) {
      eta <- draws[, i, , drop = FALSE]  
    } else {
      eta <- draws[, i, drop = FALSE]  
    }
  } else if (!is.null(draws[["nlpars"]])) {
    eta <- nonlinear_predictor(draws, i = i)
  } else if (!is.null(draws[["mv"]])) {
    eta <- linear_predictor_mv(draws, i = i)
  } else {
    eta <- linear_predictor(draws, i = i)
  }
  eta
}
