% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quantreg-rq-tidiers.R
\name{augment.rq}
\alias{augment.rq}
\title{Augment data with information from a(n) rq object}
\usage{
\method{augment}{rq}(x, data = model.frame(x), newdata = NULL, ...)
}
\arguments{
\item{x}{An \code{rq} object returned from \code{\link[quantreg:rq]{quantreg::rq()}}.}

\item{data}{A \link[base:data.frame]{base::data.frame} or \code{\link[tibble:tibble]{tibble::tibble()}} containing the original
data that was used to produce the object \code{x}. Defaults to
\code{stats::model.frame(x)} so that \code{augment(my_fit)} returns the augmented
original data. \strong{Do not} pass new data to the \code{data} argument.
Augment will report information such as influence and cooks distance for
data passed to the \code{data} argument. These measures are only defined for
the original training data.}

\item{newdata}{A \code{\link[base:data.frame]{base::data.frame()}} or \code{\link[tibble:tibble]{tibble::tibble()}} containing all
the original predictors used to create \code{x}. Defaults to \code{NULL}, indicating
that nothing has been passed to \code{newdata}. If \code{newdata} is specified,
the \code{data} argument will be ignored.}

\item{...}{
  Arguments passed on to \code{\link[quantreg:predict.rq]{quantreg::predict.rq}}
  \describe{
    \item{\code{object}}{ object of class rq or rqs or rq.process produced by \code{rq} }
    \item{\code{interval}}{type of interval desired:  default is  'none', when set to
    'confidence' the function returns a matrix predictions with point predictions
    for each of the 'newdata' points as well as lower and upper confidence limits.}
    \item{\code{level}}{converage probability for the 'confidence' intervals.}
    \item{\code{type}}{For \code{predict.rq}, the  method for 'confidence' intervals, if desired. 
    If 'percentile' then one of the bootstrap methods is used to generate percentile 
    intervals for each prediction, if 'direct' then a version of the Portnoy and Zhou 
    (1998) method is used, and otherwise an estimated covariance matrix for the parameter
    estimates is used.  Further arguments to determine the choice of bootstrap
    method or covariance matrix estimate can be passed via the \dots argument.
    For \code{predict.rqs} and \code{predict.rq.process} when \code{stepfun = TRUE},
    \code{type} is "Qhat", "Fhat" or "fhat" depending  on whether the user would
    like to have estimates of the conditional quantile, distribution or density  functions
    respectively.  As noted below the two former estimates can be monotonized with the 
    function \code{rearrange}.  When the "fhat" option is invoked, a list of conditional
    density functions is returned based on Silverman's adaptive kernel method as
    implemented in \code{akj} and \code{approxfun}.}
    \item{\code{na.action}}{ function determining what should be done with missing values
    in 'newdata'.  The default is to predict 'NA'.}
  }}
}
\description{
Augment accepts a model object and a dataset and adds
information about each observation in the dataset. Most commonly, this
includes predicted values in the \code{.fitted} column, residuals in the
\code{.resid} column, and standard errors for the fitted values in a \code{.se.fit}
column. New columns always begin with a \code{.} prefix to avoid overwriting
columns in the original dataset.

Users may pass data to augment via either the \code{data} argument or the
\code{newdata} argument. If the user passes data to the \code{data} argument,
it \strong{must} be exactly the data that was used to fit the model
object. Pass datasets to \code{newdata} to augment data that was not used
during model fitting. This still requires that at least all predictor
variable columns used to fit the model are present. If the original outcome
variable used to fit the model is not included in \code{newdata}, then no
\code{.resid} column will be included in the output.

Augment will often behave differently depending on whether \code{data} or
\code{newdata} is given. This is because there is often information
associated with training observations (such as influences or related)
measures that is not meaningfully defined for new observations.

For convenience, many augment methods provide default \code{data} arguments,
so that \code{augment(fit)} will return the augmented training data. In these
cases, augment tries to reconstruct the original data based on the model
object with varying degrees of success.

The augmented dataset is always returned as a \link[tibble:tibble]{tibble::tibble} with the
\strong{same number of rows} as the passed dataset. This means that the
passed data must be coercible to a tibble. At this time, tibbles do not
support matrix-columns. This means you should not specify a matrix
of covariates in a model formula during the original model fitting
process, and that \code{\link[splines:ns]{splines::ns()}}, \code{\link[stats:poly]{stats::poly()}} and
\code{\link[survival:Surv]{survival::Surv()}} objects are not supported in input data. If you
encounter errors, try explicitly passing a tibble, or fitting the original
model on data in a tibble.

We are in the process of defining behaviors for models fit with various
\code{na.action} arguments, but make no guarantees about behavior when data is
missing at this time.
}
\details{
Depending on the arguments passed on to \code{predict.rq} via \code{...},
a confidence interval is also calculated on the fitted values resulting in
columns \code{.lower} and \code{.upper}. Does not provide confidence
intervals when data is specified via the \code{newdata} argument.
}
\examples{
\dontshow{if (rlang::is_installed("quantreg")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

# load modeling library and data
library(quantreg)

data(stackloss)

# median (l1) regression fit for the stackloss data. 
mod1 <- rq(stack.loss ~ stack.x, .5)  

# weighted sample median
mod2 <- rq(rnorm(50) ~ 1, weights = runif(50))

# summarize model fit with tidiers
tidy(mod1)
glance(mod1)
augment(mod1)

tidy(mod2)
glance(mod2)
augment(mod2)

# varying tau to generate an rqs object
mod3 <- rq(stack.loss ~ stack.x, tau = c(.25, .5))

tidy(mod3)
augment(mod3)

# glance cannot handle rqs objects like `mod3`--use a purrr 
# `map`-based workflow instead
\dontshow{\}) # examplesIf}
}
\seealso{
\link{augment}, \code{\link[quantreg:rq]{quantreg::rq()}}, \code{\link[quantreg:predict.rq]{quantreg::predict.rq()}}

Other quantreg tidiers: 
\code{\link{augment.nlrq}()},
\code{\link{augment.rqs}()},
\code{\link{glance.nlrq}()},
\code{\link{glance.rq}()},
\code{\link{tidy.nlrq}()},
\code{\link{tidy.rqs}()},
\code{\link{tidy.rq}()}
}
\concept{quantreg tidiers}
\value{
A \code{\link[tibble:tibble]{tibble::tibble()}} with columns:
  \item{.fitted}{Fitted or predicted value.}
  \item{.resid}{The difference between observed and fitted values.}
  \item{.tau}{Quantile.}

}
