% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ccRemover.R
\name{ccRemover}
\alias{ccRemover}
\title{Removes the effect of the cell-cycle}
\usage{
ccRemover(dat, cutoff = 3, max_it = 4, nboot = 200, ntop = 10,
  bar = TRUE)
}
\arguments{
\item{dat}{A list containing a data frame , \code{x}, that contains gene expression
measurements with each column representing a sample and each row
representing a gene and a logical vector, \code{if_cc}, that indicates
which of the genes/rows are related to the cell-cycle or factor of interest.

It is recommended that the elements of x are log-transformed and centered
for each gene. For example if \code{x} contains TPM measurements then we
suggest the following two-steps:
Step 1: \code{dat$x <- log(dat$x + 1)}
Step 2: \code{dat$x} - rowMeans(dat$x)
ccRemover requires that the samples have been properly normalized for
sequencing depth and we recommend doing so prior to applying the above steps.

The \code{if_cc} vector must be the same length as the number of rows in
\code{x} and have elements equal to \code{TRUE} for genes which are related
to the cell-cycle and and elements equal to \code{FALSE} for genes which
are unrelated to the cell-cycle.}

\item{cutoff}{The significance cutoff for identifying sources of variation
related to the cell-cycle. The default value is 3, which roughly corresponds
to a p-value of 0.01.}

\item{max_it}{The maximum number of iterations for the algorithm. The
default value is 4.}

\item{nboot}{The number of bootstrap repititions to be carried out on each
iteration to determine the significance of each component.}

\item{ntop}{The number of components considered tested at each iteration as
cell-cycle effects. The default value if 10}

\item{bar}{Whether to display a progress bar or not. The progress bar will
not work in R-markdown enviornments so this option may be turned off. The
default value is \code{TRUE}.}
}
\value{
A data matrix with the effects of the cell-cycle removed.
}
\description{
\code{ccRemover} returns a data matrix with the effects of the cell-cycle
removed.
}
\details{
Implements the algorithm described in Barron, M. & Li, J.
"Identifying and removing the cell-cycle effect from scRNA-Sequencing data"
(2016), Scientific Reports. This function takes a normalized,
log-transformed and centered matrix of scRNA-seq  expression data
and a list of genes which are known to be related to the cell-cycle effect.
It then captures the main sources of variation in the data and determines
which of these are related to the cell-cycle before removing those that are.
Please see the original manuscript for further details.
}
\examples{
set.seed(10)
# Load in example data
data(t.cell_data)
head(t.cell_data[,1:5])
# Center data and select small sample for example
t_cell_data_cen <- t(scale(t(t.cell_data[,1:20]), center=TRUE, scale=FALSE))
# Extract gene names
gene_names <- rownames(t_cell_data_cen)
# Determine which genes are annotated to the cell-cycle
cell_cycle_gene_indices <- gene_indexer(gene_names,
species = "mouse", name_type = "symbol")
# Create "if_cc" vector
if_cc <- rep(FALSE,nrow(t_cell_data_cen))
if_cc[cell_cycle_gene_indices] <- TRUE
# Move data into list
dat <- list(x=t_cell_data_cen, if_cc=if_cc)
# Run ccRemover
\dontrun{
 xhat <- ccRemover(dat, cutoff = 3, max_it = 4, nboot = 200, ntop = 10)
}
# Run ccRemover with reduced bootstrap repetitions for example only
xhat <- ccRemover(dat, cutoff = 3, max_it = 4, nboot = 20, ntop = 10)
head(xhat[,1:5])
# Run ccRemover with more compoents considered
\dontrun{
xhat <- ccRemover(dat, cutoff = 3, max_it = 4, nboot = 200, ntop = 15)
 }

}
