% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confidence_intervals.R
\name{conf_region}
\alias{conf_region}
\title{Two-dimensional confidence regions}
\usage{
conf_region(object, which_pars = NULL, range1 = c(NA, NA),
  range2 = c(NA, NA), conf = 95, mult = 2, num = c(10, 10),
  type = c("vertical", "cholesky", "spectral", "none"), ...)
}
\arguments{
\item{object}{An object of class \code{"chandwich"} returned by
\code{adjust_loglik}.}

\item{which_pars}{A vector of length 2 specifying the 2 (unfixed)
  parameters for which confidence region is required.
  Can be either a numeric vector, specifying indices of the components
  of the \strong{full} parameter vector, or a character vector of
  parameter names, which must be a subset of those supplied in
  \code{par_names} in the call to \code{\link{adjust_loglik}} that
  produced \code{object}.

  \code{which_pars} must not have any parameters in common with
  \code{attr(object, "fixed_pars")}.  \code{which_pars} must not contain
  all of the unfixed parameters, i.e. there is no point in profiling over
  all the unfixed parameters.

  If \code{which_pars} is not supplied but the current model has exactly
  two free parameters, i.e. \code{attr(object, "p_current") = 2} then
  \code{which_pars} is set to \code{attr(object, "free_pars") = 2}.}

\item{range1, range2}{Numeric vectors of length 2.  Respective ranges (of
the form \code{lower, upper}) of values of \code{which_pars[1]} and
\code{which_pars[2]} over which to profile.
Missing values in \code{range1} and/or \code{range2} are
filled in using \code{conf} and \code{mult}.  See below for details.}

\item{conf}{A numeric scalar in (0, 100).  The highest confidence level
of interest. This is only relevant if \code{lower} and \code{upper} are
not supplied.  In that event \code{conf} is used, in combination with
\code{mult}, to try to set up the grid of parameter values to include
the largest confidence region of interest.}

\item{mult}{A numeric vector of length 1 or the same length as
\code{which_pars}.
The search for the profile loglikelihood-based confidence limits is
conducted over the corresponding symmetric confidence intervals
(based on approximate normal theory), extended
by a factor of the corresponding component of \code{mult}.}

\item{num}{A numeric vector of length 1 or 2.  The numbers of values at which
to evaluate the profile loglikelihood either side of the MLE.
\code{num[i]} relates to \code{which_pars[i]}.  If \code{num} has length
1 then \code{num} is replicated to have length 2.}

\item{type}{A character scalar.  The argument \code{type} to the function
returned by \code{\link{adjust_loglik}}, that is, the type of adjustment
made to the independence loglikelihood function.}

\item{...}{Further arguments to be passed to \code{\link[stats]{optim}}.
These may include \code{gr}, \code{method}, \code{lower}, \code{upper}
or \code{control}.  Any arguments that are not appropriate for
\code{\link[stats]{optim}}, i.e. not in
\code{methods::formalArgs(stats::optim)},
will be removed without warning.}
}
\value{
An object of class "confreg", a list with components
    \item{grid1, grid2}{Numeric vectors.   Respective values of
      \code{which_pars[1]} and \code{which_pars[2]} in the grid over which
      the (profile) loglikelihood is evaluated. }
    \item{max_loglik}{A numeric scalar.  The value value of
      the loglikelihood at its maximum.}
    \item{prof_loglik}{An 2 \code{num} + 1 by 2 \code{num} + 1
      numeric matrix containing the values of the (profile) loglikelihood.}
    \item{type}{A character scalar. The input \code{type}.}
    \item{which_pars}{A numeric or character vector.  The input
      \code{which_pars}.  If the \code{which_pars} was numeric then
      it is supplemented by the parameter names, if these are available
      in \code{object}.}
    \item{name}{A character scalar. The name of the model,
      stored in \code{attr(object, "name")}.}
}
\description{
Calculates the (profile, if necessary) loglikelihood for a pair of
parameters from which confidence regions can be plotted using
\code{\link{plot.confreg}}.
}
\examples{
# -------------------------- GEV model, owtemps data -----------------------
# ------------ following Section 5.2 of Chandler and Bate (2007) -----------

gev_loglik <- function(pars, data) {
  o_pars <- pars[c(1, 3, 5)] + pars[c(2, 4, 6)]
  w_pars <- pars[c(1, 3, 5)] - pars[c(2, 4, 6)]
  if (o_pars[2] <= 0 | w_pars[2] <= 0) return(-Inf)
  o_data <- data[, "Oxford"]
  w_data <- data[, "Worthing"]
  check <- 1 + o_pars[3] * (o_data - o_pars[1]) / o_pars[2]
  if (any(check <= 0)) return(-Inf)
  check <- 1 + w_pars[3] * (w_data - w_pars[1]) / w_pars[2]
  if (any(check <= 0)) return(-Inf)
  o_loglik <- log_gev(o_data, o_pars[1], o_pars[2], o_pars[3])
  w_loglik <- log_gev(w_data, w_pars[1], w_pars[2], w_pars[3])
  return(o_loglik + w_loglik)
}

# Initial estimates (method of moments for the Gumbel case)
sigma <- as.numeric(sqrt(6 * diag(var(owtemps))) / pi)
mu <- as.numeric(colMeans(owtemps) - 0.57722 * sigma)
init <- c(mean(mu), -diff(mu) / 2, mean(sigma), -diff(sigma) / 2, 0, 0)

# Log-likelihood adjustment of the full model
par_names <- c("mu[0]", "mu[1]", "sigma[0]", "sigma[1]", "xi[0]", "xi[1]")
large <- adjust_loglik(gev_loglik, data = owtemps, init = init,
                       par_names = par_names)

\dontrun{
# Plots like those in Figure 4 of Chandler and Bate (2007)
# (a)
which_pars <- c("mu[0]", "mu[1]")
reg_1 <- conf_region(large, which_pars = which_pars)
reg_none_1 <- conf_region(large, which_pars = which_pars, type = "none")
plot(reg_1, reg_none_1)
# (b)
which_pars <- c("sigma[0]", "sigma[1]")
reg_2 <- conf_region(large, which_pars = which_pars)
reg_none_2 <- conf_region(large, which_pars = which_pars, type = "none")
plot(reg_2, reg_none_2)
# (c)
# Note: the naive and bivariate model contours are the reversed in the paper
which_pars <- c("sigma[0]", "xi[0]")
reg_3 <- conf_region(large, which_pars = which_pars)
reg_none_3 <- conf_region(large, which_pars = which_pars, type = "none")
plot(reg_3, reg_none_3)
}

# --------- Misspecified Poisson model for negative binomial data ----------

# ... following Section 5.1 of the "Object-Oriented Computation of Sandwich
# Estimators" vignette of the sandwich package
# https://cran.r-project.org/web/packages/sandwich/vignettes/sandwich-OOP.pdf

# Simulate data
set.seed(123)
x <- rnorm(250)
y <- rnbinom(250, mu = exp(1 + x), size = 1)
# Fit misspecified Poisson model
fm_pois <- glm(y ~ x + I(x^2), family = poisson)
summary(fm_pois)$coefficients

# Contributions to the independence loglikelihood
pois_glm_loglik <- function(pars, y, x) {
  log_mu <- pars[1] + pars[2] * x + pars[3] * x ^ 2
  return(dpois(y, lambda = exp(log_mu), log = TRUE))
}
pars <- c("alpha", "beta", "gamma")
# Linear model (gamma fixed at 0)
pois_lin <- adjust_loglik(pois_glm_loglik, y = y, x = x, par_names = pars,
                          fixed_pars = "gamma")
pois_vertical <- conf_region(pois_lin)
pois_none <- conf_region(pois_lin, type = "none")
plot(pois_none, pois_vertical, conf = c(50, 75, 95, 99), col = 2:1, lwd = 2,
     lty = 1)
}
\seealso{
\code{\link{adjust_loglik}} to adjust a user-supplied
  loglikelihood function.

\code{\link{conf_intervals}} for confidence intervals for
  individual parameters.

\code{\link{compare_models}} to compare nested models using an
  (adjusted) likelihood ratio test.

\code{\link{plot.confreg}}.
}
