% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_probs.R
\name{add_probs}
\alias{add_probs}
\title{Add Regression Probabilities to Data Frames}
\usage{
add_probs(tb, fit, q, name = NULL, yhatName = "pred", comparison, ...)
}
\arguments{
\item{tb}{A tibble or data frame of new data.}

\item{fit}{An object of class \code{lm}, \code{glm}, or
\code{lmerMod}. Predictions are made with this object.}

\item{q}{A real number. A quantile of the conditional response
distribution.}

\item{name}{\code{NULL} or character vector of length one. If
\code{NULL}, probabilities automatically will be named by
\code{add_probs}, otherwise, the probabilities will be named
\code{name} in the returned tibble.}

\item{yhatName}{A character vector of length one. Names of the}

\item{comparison}{A string. If \code{comparison = "<"}, then
\eqn{Pr(Y|x < q)} is calculated for each observation in
\code{tb}. Default is "<". Must be "<" or ">" for objects of
class \code{lm} or \code{lmerMod}. If \code{fit} is a
\code{glm}, then \code{comparison} also may be \code{"<="} ,
\code{">="} , or \code{"="}.}

\item{...}{Additional arguments}
}
\value{
A tibble, \code{tb}, with predicted values and
    probabilities attached.
}
\description{
This is a generic function to append response level probabilities
to a data frame. A response level probability (conditioned on the
model and covariates), such as \eqn{Pr(Response|Covariates < 10)},
is generated for the fitted value of each observation in
\code{tb}. These probabilities are then appended to \code{tb} and
returned to the user as a tibble.
}
\details{
For more specific information about the arguments that are useful
in each method, consult:

\itemize{
  \item \code{\link{add_probs.lm}} for linear regression response probabilities
  \item \code{\link{add_probs.glm}} for generalized linear regression response probabilities
  \item \code{\link{add_probs.lmerMod}} for linear mixed models response probabilities
  \item \code{\link{add_probs.glmerMod}} for generalized linear mixed model response probabilities
  \item \code{\link{add_probs.survreg}} for accelerated failure time model response probabilities
}

Note: Except in \code{add_probs.survreg}, the probabilities
calculated by \code{add_probs} are on the distribution of
\eqn{Y|x}, not \eqn{E[Y|x]}. That is, they use the same
distribution from which a prediction interval is determined, not
the distribution that determines a confidence
interval. \code{add_probs.survreg} is an exception to this pattern
so that users of accelerated failure time models can obtain
estimates of the survivor function.
}
\examples{
# Define a model
fit <- lm(dist ~ speed, data = cars)

# Calculate the probability that the probability that a new
# dist is less than 20 (Given the model).
add_probs(cars, fit, q = 20)

# Calculate the probability that a new
# dist is greater than 20 (Given the model).
add_probs(cars, fit, q = 20, comparison = ">")

# Try a different model fit.
fit2 <- glm(dist ~ speed, family = "poisson", data = cars)
add_probs(cars, fit2, q = 20)

# Try a different model fit.
fit3 <- lme4::lmer(Reaction ~ Days + (1|Subject), data = lme4::sleepstudy)
add_probs(lme4::sleepstudy, fit3, q = 300, type = "parametric")

# As above, but do not condition on the random effects.
add_probs(lme4::sleepstudy, fit3, q = 300, type = "parametric", includeRanef = FALSE)

}
\seealso{
\code{\link{add_ci}} for confidence intervals,
    \code{\link{add_quantile}} for response level quantiles, and
    \code{\link{add_pi}} for prediction intervals.
}
