% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convSigCt.R
\name{convSigCt}
\alias{convSigCt}
\title{Computes and conducts significance tests on Ct-metric scores for putatively convergent tips (or groups of tips) given a set of user provided phenotypic characters and a time calibrated phylogeny.}
\usage{
convSigCt(phy, traits, focaltaxa, groups = NULL, nsim = 1000, ...)
}
\arguments{
\item{phy}{The time calibrated phylogeny of interest in phylo format}

\item{traits}{a matrix of numeric phenotypic traits with rownames matching tip labels of phy}

\item{focaltaxa}{a vector of tip labels for the putatively convergent taxa to be compared}

\item{groups}{an optional vector of groups with names matching focaltaxa. Indicates the group identity of all putatively convergent taxa and limits Ct measures to intergroup comparisons only}

\item{nsim}{number of simulated (Brownian motion) datasets used to build the null distribution}

\item{...}{optional arguments to be passed to calcConv. If convservative is TRUE, Dmax.t is restricted to before the origin of the oldest lineage in each pairwise comparison of the focal taxa. The origin of convergent lineages is taken as the most recent common ancestors of each user defined group. Where these groups include a single tip, the parent node of its subtending branch is used. Requires user to provide groups. If VERBOSE is TRUE, model information will be printed during computation, including time limits imposed on Dmax.t if the conservative option is chosen.}
}
\value{
A list of the following components:

pvals a matrix containing Ct1 - Ct4 and p-values from significance tests for each

meas.Cmat a matrix of Ct values for each pairwise comparison of focaltaxa

meas.path a list of dataframes, one per pairwise comparison of focaltaxa, each containing information from all timepoint measurements of the two putatively convergent lineages. These provide the nodes at which comparisons were drawn, the evolutionary path along which that node fell (i.e., leading to one of two tips), the node height, reconstructed ancestral states at that node for each phenotypic trait, reconstructed ancestral values for each trait along the opposite path, and the phenotypic distance between the two lineages at that point.

sim.avg average Ct values from all pairwise comparisons between focaltaxa using simulated Brownian motion traits, number of columns corresponds to the user provided number of simulations

sim.path a list of dataframes as in meas.path, but obtained using simulated data. Length of object determined by number of pairwise comparisons multiplied by the number of simulated datasets.

grp.mean a matrix of Ct-metrics summarized for inter-group comparisons, returned only if user defined groups were specified. Provides overall results matching those reported in "mean", results for each unique inter-group comparison, and results averaged with equal weight given to each unique inter-group comparison (i.e., ignoring differences in the number of tips in each group).

grp.pvals a matrix of p-values associated with Ct values in grp.mean object. Returned only if user defined groups were specified.
}
\description{
convSigCt Computes and conducts significance tests on Ct-metric scores for putatively convergent tips (or groups of tips) given a set of user provided phenotypic characters and a time calibrated phylogeny.
}
\details{
Function incorporates the optimizations introduced by Zelditch et al. (2017), which significantly improve runtimes

Reconstructions part way along branches are obtained using equation [2] of Felsenstein (1985), following code modified from the phytools (Revell, 2012) function contMap
}
\examples{

\donttest{# create time calibrated tree
mytree<-rtree(50)
mycalibration <- makeChronosCalib(mytree, node="root", age.max=50)
phy <- chronos(mytree, calibration = mycalibration, control = chronos.control() )
class(phy)<-"phylo"

# create three normally distributed phenotypic traits
traits <- cbind(rnorm(Ntip(phy)),rnorm(Ntip(phy)),rnorm(Ntip(phy)))
rownames(traits) <- phy$tip.label
focaltaxa <- sample(phy$tip.label, 5)

system.time(run2 <- convSigCt(phy, traits, focaltaxa, nsim=10))}
}
\references{
Grossnickle DM, Brightly WH, Weaver LN, Stanchak KE, Roston RA, Pevsner SK, Stayton CT, Polly PD, Law CJ. 2022. A cautionary note on quantitative measures of phenotypic convergence. in revision
Zelditch ML, Ye J, Mitchell JS, Swiderski DL. 2017. Rare ecomorphological convergence on a complex adaptive landscape: Body size and diet mediate evolution of jaw shape in squirrels (Sciuridae). Evolution 71: 633-649
Stayton CT. 2015. The definition, recognition, and interpretation of convergent evolution and two new measures for quantifying and assessing the significance of convergence. Evolution 69(8): 2140-2153.
Revell, L. J. (2012) phytools: An R package for phylogenetic comparative biology (and other things). Methods Ecol. Evol., 3, 217-223.
Felsenstein, J. 1985. Phylogenies and the comparative method. American Naturalist, 125, 1-15.
}
