% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/STAR_Bayesian.R
\name{spline_star}
\alias{spline_star}
\title{Estimation for Bayesian STAR spline regression}
\usage{
spline_star(
  y,
  tau = NULL,
  transformation = "np",
  y_max = Inf,
  psi = NULL,
  approx_Fz = FALSE,
  approx_Fy = FALSE,
  nsave = 1000,
  use_MCMC = TRUE,
  nburn = 1000,
  nskip = 0,
  verbose = TRUE,
  method_sigma = "mle"
)
}
\arguments{
\item{y}{\code{n x 1} vector of observed counts}

\item{tau}{\code{n x 1} vector of observation points; if NULL, assume equally-spaced on [0,1]}

\item{transformation}{transformation to use for the latent data; must be one of
\itemize{
\item "identity" (identity transformation)
\item "log" (log transformation)
\item "sqrt" (square root transformation)
\item "bnp" (Bayesian nonparametric transformation using the Bayesian bootstrap)
\item "np" (nonparametric transformation estimated from empirical CDF)
\item "pois" (transformation for moment-matched marginal Poisson CDF)
\item "neg-bin" (transformation for moment-matched marginal Negative Binomial CDF)
}}

\item{y_max}{a fixed and known upper bound for all observations; default is \code{Inf}}

\item{psi}{prior variance (1/smoothing parameter); if NULL, update in MCMC}

\item{approx_Fz}{logical; in BNP transformation, apply a (fast and stable)
normal approximation for the marginal CDF of the latent data}

\item{approx_Fy}{logical; in BNP transformation, approximate
the marginal CDF of \code{y} using the empirical CDF}

\item{nsave}{number of MCMC iterations to save (or number of Monte Carlo simulations)}

\item{use_MCMC}{logical; whether to run Gibbs sampler or Monte Carlo (default is TRUE)}

\item{nburn}{number of MCMC iterations to discard}

\item{nskip}{number of MCMC iterations to skip between saving iterations,
i.e., save every (nskip + 1)th draw}

\item{verbose}{logical; if TRUE, print time remaining}

\item{method_sigma}{method to estimate the latent data standard deviation (only applicable if \code{use_MCMC=FALSE});
must be one of
\itemize{
\item "mle" use the MLE from the STAR EM algorithm (default)
\item "mmle" use the marginal MLE (Note: slower!)
}}
}
\value{
a list with the following elements:
\itemize{
\item \code{post_ytilde}: \code{nsave x n} samples
from the posterior predictive distribution at the observation points \code{tau}
\item \code{marg_like}: the marginal likelihood (only if \code{use_MCMC=FALSE}; otherwise NULL)
}
}
\description{
Compute samples from the predictive
distributions of a STAR spline regression model using either a Gibbs sampling approach
or exact Monte Carlo sampling (default is Gibbs sampling which scales better for large n)
}
\details{
STAR defines a count-valued probability model by
(1) specifying a Gaussian model for continuous *latent* data and
(2) connecting the latent data to the observed data via a
*transformation and rounding* operation. Here, the continuous
latent data model is a spline regression.

There are several options for the transformation. First, the transformation
can belong to the *Box-Cox* family, which includes the known transformations
'identity', 'log', and 'sqrt'. Second, the transformation
can be estimated (before model fitting) using the empirical distribution of the
data \code{y}. Options in this case include the empirical cumulative
distribution function (CDF), which is fully nonparametric ('np'), or the parametric
alternatives based on Poisson ('pois') or Negative-Binomial ('neg-bin')
distributions. For the parametric distributions, the parameters of the distribution
are estimated using moments (means and variances) of \code{y}. The distribution-based
transformations approximately preserve the mean and variance of the count data \code{y}
on the latent data scale, which lends interpretability to the model parameters.
Lastly, the transformation can be modeled using the Bayesian bootstrap ('bnp'),
which is a Bayesian nonparametric model and incorporates the uncertainty
about the transformation into posterior and predictive inference.
}
\note{
For the 'bnp' transformation (without the \code{Fy} approximation),
there are numerical stability issues when \code{psi} is modeled as unknown.
In this case, it is better to fix \code{psi} at some positive number.
}
\examples{
# Simulate some data:
n = 100
tau = seq(0,1, length.out = n)
y = round_floor(exp(1 + rnorm(n)/4 + poly(tau, 4)\%*\%rnorm(n=4, sd = 4:1)))

# Sample from the predictive distribution of a STAR spline model:
fit = spline_star(y = y, tau = tau)

# Compute 90\% prediction intervals:
pi_y = t(apply(fit$post_ytilde, 2, quantile, c(0.05, .95)))

# Plot the results: intervals, median, and smoothed mean
plot(tau, y, ylim = range(pi_y, y))
polygon(c(tau, rev(tau)),c(pi_y[,2], rev(pi_y[,1])),col='gray', border=NA)
lines(tau, apply(fit$post_ytilde, 2, median), lwd=5, col ='black')
lines(tau, smooth.spline(tau, apply(fit$post_ytilde, 2, mean))$y, lwd=5, col='blue')
lines(tau, y, type='p')

}
