test_that("schedule validate empty", {
  x <- crew_schedule()
  expect_silent(x$validate())
})

test_that("schedule validate full", {
  x <- crew_schedule()
  x$start()
  x$head <- "id"
  x$until <- nanonext::mclock()
  expect_silent(x$validate())
})

test_that("schedule start", {
  x <- crew_schedule()
  expect_null(x$pushed)
  expect_null(x$collected)
  expect_null(x$pushes)
  x$start()
  expect_true(is.environment(x$pushed))
  expect_true(is.environment(x$collected))
  expect_equal(x$pushes, 0L)
})

test_that("schedule push", {
  x <- crew_schedule()
  x$start()
  expect_equal(length(x$pushed), 0L)
  expect_equal(length(x$collected), 0L)
  expect_equal(x$pushes, 0L)
  expect_null(x$head)
  expect_null(x$until)
  x$push(task = crew_null)
  expect_equal(x$pushes, 1L)
  expect_equal(length(x$pushed), 1L)
  expect_equal(length(x$collected), 0L)
  expect_null(x$head)
  expect_null(x$until)
  expect_true(is_crew_null(x$pushed[["1"]]))
})

test_that("schedule list", {
  x <- crew_schedule()
  x$start()
  x$push(task = list(data = "contents"))
  x$collect(throttle = FALSE)
  expect_equal(x$list()[[1L]], "contents")
})

test_that("schedule throttle", {
  x <- crew_schedule(seconds_interval = 9999)
  x$start()
  expect_null(x$until)
  expect_true(x$throttle())
  expect_true(is.numeric(x$until))
  x$until <- Inf
  expect_true(x$throttle())
  x$until <- -Inf
  expect_false(x$throttle())
  expect_null(x$until)
  expect_true(x$throttle())
  expect_true(is.numeric(x$until))
})

test_that("schedule collect without throttling", {
  x <- crew_schedule()
  x$start()
  x$push(task = crew_null)
  x$collect(throttle = FALSE)
  expect_equal(length(x$pushed), 0L)
  expect_equal(length(x$collected), 1L)
  expect_true(is_crew_null(x$collected[["1"]]$task))
  expect_null(x$collected[["1"]]$head)
  expect_null(x$until)
  expect_equal(x$head, "1")
})

test_that("schedule collect with throttling and short interval", {
  skip_on_cran()
  x <- crew_schedule(seconds_interval = 0.01)
  x$start()
  x$push(task = crew_null)
  x$collect(throttle = TRUE)
  expect_equal(length(x$pushed), 1L)
  expect_equal(length(x$collected), 0L)
  expect_null(x$head)
  expect_false(is.null(x$until))
  expect_true(is_crew_null(x$pushed[["1"]]))
  Sys.sleep(0.1)
  x$collect(throttle = TRUE)
  expect_equal(length(x$pushed), 0L)
  expect_equal(length(x$collected), 1L)
  expect_true(is_crew_null(x$collected[["1"]]$task))
  expect_null(x$until)
  expect_equal(x$head, "1")
})

test_that("schedule collect with throttling and long interval", {
  skip_on_cran()
  x <- crew_schedule(seconds_interval = 9999)
  x$start()
  x$push(task = crew_null)
  for (index in seq_len(3L)) {
    x$collect(throttle = TRUE)
    expect_equal(length(x$pushed), 1L)
    expect_equal(length(x$collected), 0L)
    expect_null(x$head)
    expect_false(is.null(x$until))
    expect_true(is_crew_null(x$pushed[["1"]]))
    Sys.sleep(0.1)
  }
})

test_that("schedule pop", {
  x <- crew_schedule()
  expect_null(x$pop())
  expect_null(x$head)
  x$start()
  expect_null(x$pop())
  expect_null(x$head)
  x$push(task = crew_null)
  expect_null(x$head)
  expect_null(x$pop())
  x$collect(throttle = FALSE)
  expect_equal(x$head, "1")
  task <- x$pop()
  expect_true(is_crew_null(task))
  expect_null(x$head)
})

test_that("schedule collected queue", {
  x <- crew_schedule()
  x$start()
  numbers <- as.character(seq_len(26L))
  for (number in numbers) {
    x$push(task = list(number = number))
  }
  expect_equal(x$pushes, length(numbers))
  x$collect(throttle = FALSE)
  expect_true(nzchar(x$head))
  results <- character(0L)
  while (length(x$pushed) > 0L || length(x$collected) > 0L) {
    expect_true(x$head %in% setdiff(numbers, results))
    task <- x$pop()
    results <- c(results, task$number)
  }
  expect_null(x$head)
  expect_null(x$pop())
  expect_equal(sort(results), sort(numbers))
})

test_that("schedule empty", {
  x <- crew_schedule()
  expect_true(x$empty())
  x$start()
  x$push(task = crew_null)
  expect_false(x$empty())
  x$collect(throttle = FALSE)
  expect_false(x$empty())
  x$pop()
  expect_true(x$empty())
})

test_that("schedule nonempty", {
  x <- crew_schedule()
  expect_false(x$nonempty())
  x$start()
  x$push(task = crew_null)
  expect_true(x$nonempty())
  x$collect(throttle = FALSE)
  expect_true(x$nonempty())
  x$pop()
  expect_false(x$nonempty())
})

test_that("schedule collected_all", {
  x <- crew_schedule()
  expect_true(x$collected_mode(mode = "all"))
  x$start()
  expect_true(x$collected_mode(mode = "all"))
  x$push(crew_null)
  expect_false(x$collected_mode(mode = "all"))
  x$push(crew_null)
  expect_false(x$collected_mode(mode = "all"))
  x$collect()
  expect_true(x$collected_mode(mode = "all"))
  x$push(crew_null)
  expect_false(x$collected_mode(mode = "all"))
  x$collect()
  expect_true(x$collected_mode(mode = "all"))
  while (x$nonempty()) {
    x$pop()
    expect_true(x$collected_mode(mode = "all"))
  }
})

test_that("schedule collected_one", {
  x <- crew_schedule()
  expect_true(x$collected_mode(mode = "one"))
  x$start()
  expect_true(x$collected_mode(mode = "one"))
  x$push(crew_null)
  expect_false(x$collected_mode(mode = "one"))
  x$push(crew_null)
  expect_false(x$collected_mode(mode = "one"))
  x$collect()
  expect_true(x$collected_mode(mode = "one"))
  x$push(crew_null)
  expect_true(x$collected_mode(mode = "one"))
  x$collect()
  expect_true(x$collected_mode(mode = "one"))
  while (x$nonempty()) {
    x$pop()
    expect_true(x$collected_mode(mode = "one"))
  }
})

test_that("schedule summary", {
  x <- crew_schedule()
  expect_null(x$summary())
  x$start()
  expect_true(tibble::is_tibble(x$summary()))
  expect_equal(nrow(x$summary()), 1L)
  expect_equal(sort(colnames(x$summary())), sort(c("pushed", "collected")))
  expect_equal(x$summary()$pushed, 0L)
  expect_equal(x$summary()$collected, 0L)
  x$push(task = crew_null)
  expect_equal(x$summary()$pushed, 1L)
  expect_equal(x$summary()$collected, 0L)
  x$collect(throttle = FALSE)
  expect_equal(x$summary()$pushed, 0L)
  expect_equal(x$summary()$collected, 1L)
  x$pop()
  expect_equal(x$summary()$pushed, 0L)
  expect_equal(x$summary()$collected, 0L)
})
