% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crosstable.R
\name{crosstable}
\alias{crosstable}
\title{Easily describe datasets}
\usage{
crosstable(
  data,
  cols = everything(),
  ...,
  by = NULL,
  total = c("none", "row", "column", "both"),
  percent_pattern = "{n} ({p_row})",
  percent_digits = 2,
  num_digits = 1,
  showNA = c("ifany", "always", "no"),
  label = TRUE,
  funs = c(` ` = cross_summary),
  funs_arg = list(),
  cor_method = c("pearson", "kendall", "spearman"),
  drop_levels = FALSE,
  unique_numeric = 3,
  date_format = NULL,
  times = NULL,
  followup = FALSE,
  test = FALSE,
  test_args = crosstable_test_args(),
  effect = FALSE,
  effect_args = crosstable_effect_args(),
  margin = deprecated(),
  .vars = deprecated()
)
}
\arguments{
\item{data}{A data.frame}

\item{cols}{<\code{\link[tidyselect:language]{tidy-select}}> Columns to describe, default to \code{everything()}. See examples or \code{vignette("crosstable-selection")} for more details.}

\item{...}{Unused. All parameters after this one must be named.}

\item{by}{The variable to group on. Character or name.}

\item{total}{one of ["none", "row", "column" or "both"] to indicate whether to add total rows and/or columns. Default to \code{none}.}

\item{percent_pattern}{Pattern used to describe proportions in categorical data. Syntax uses a \code{\link[glue:glue]{glue::glue()}} specification, see the \strong{section} below for more details. Default to \code{"{n} ({p_col})"} if \code{by} is null and \code{"{n} ({p_row})"} if it is not.}

\item{percent_digits}{Number of digits for percentages.}

\item{num_digits}{Number of digits for numeric summaries.}

\item{showNA}{Whether to show NA in categorical variables (one of \code{c("ifany", "always", "no")}, like in \code{table()}).}

\item{label}{Whether to show labels. See \code{\link[=import_labels]{import_labels()}} or \code{\link[=set_label]{set_label()}}for how to add labels to the dataset columns.}

\item{funs}{Functions to apply to numeric variables. Default to \code{\link[=cross_summary]{cross_summary()}}.}

\item{funs_arg}{Additional parameters for \code{funs}, e.g. \code{digits} (the number of decimal places) for the default \code{\link[=cross_summary]{cross_summary()}}. Ultimately, these arguments are passed to \code{\link[=format_fixed]{format_fixed()}}.}

\item{cor_method}{One of \code{c("pearson", "kendall", "spearman")} to indicate which correlation coefficient is to be used.}

\item{drop_levels}{Whether to drop unused levels of factor variables. Default to \code{TRUE}.}

\item{unique_numeric}{The number of non-missing different levels a variable should have to be considered as numeric.}

\item{date_format}{if \code{x} is a vector of Date or POSIXt, the format to apply (see \link{strptime} for formats)}

\item{times}{When using formula with \code{\link[survival:Surv]{survival::Surv()}} objects, which times to summarize.}

\item{followup}{When using formula with \code{\link[survival:Surv]{survival::Surv()}} objects, whether to display follow-up time.}

\item{test}{Whether to perform tests.}

\item{test_args}{See \code{\link{crosstable_test_args}} to override default testing behaviour.}

\item{effect}{Whether to compute a effect measure.}

\item{effect_args}{See \code{\link{crosstable_effect_args}} to override default behaviour.}

\item{margin}{Deprecated in favor of \code{percent_pattern}. One of ["row", "column", "cell", "none", or "all"]. Default to \code{row}.}

\item{.vars}{Deprecated in favor of \code{cols}.}
}
\value{
A \code{data.frame}/\code{tibble} of class \code{crosstable}
}
\description{
Generate a descriptive table of all chosen columns, as contingency tables for categorical variables and as calculation summaries for numeric variables. If the \code{by} argument points to one or several categorical variables, \code{crosstable} will output a description of all columns for each level. Otherwise, if it points to a numeric variable, \code{crosstable} will calculate correlation coefficients with all other selected numeric columns. Finally, if it points to a \code{Surv} object, \code{crosstable} will describe the survival at different times.\cr
\cr
Can be formatted as an HTML table using \code{\link[=as_flextable]{as_flextable()}}.
}
\section{About \code{percent_pattern}}{

The \code{percent_pattern} argument is very powerful but can be difficult to understand at first :
\itemize{
\item It is usually a single string that uses the glue syntax, where variables are put in curly braces (\code{{x}}).
\item Counts are expressed as \code{{n}}, \code{{n_row}}, \code{{n_col}}, and \code{{n_tot}}, and proportions as \code{{p_row}}, \code{{p_col}}, and \code{{p_cell}}, depending on the margin on which they are calculated.
\item For each variable, a version including missing values in the total is proposed as \code{{n_xxx_na}} or \code{{p_xxx_na}}.
\item For each proportion, a confidence interval is also calculated using \href{https://en.wikipedia.org/wiki/Binomial_proportion_confidence_interval#Wilson_score_interval}{Wilson score} and can be expressed as \code{{p_xxx_inf}} and \code{{p_xxx_sup}}. See examples for practical applications.
\item Alternatively, \code{percent_pattern} can be a list of characters with names \code{body}, \code{total_row}, \code{total_col}, and \code{total_all} to also control the pattern in other parts of the crosstable than the body.
}
}

\examples{
#whole table
crosstable(iris)
crosstable(mtcars)
crosstable(mtcars2)

#tidyselection, custom functions
library(dplyr)
crosstable(mtcars2, c(ends_with("t"), starts_with("c")), by=vs,
           funs=c(mean, quantile), funs_arg=list(probs=c(.25,.75)))

#margin and totals, multiple by
crosstable(mtcars2, c(disp, cyl), by=c(am, vs),
           margin=c("row", "col"), total = "both")

#predicate selection, correlation, effect calculation
crosstable(mtcars2, where(is.numeric), by=hp, effect=TRUE)

#lambda selection & statistical tests
crosstable(mtcars2, ~is.numeric(.x) && mean(.x)>50, by=vs, test=TRUE)

#Dates
mtcars2$my_date = as.Date(mtcars2$hp , origin="2010-01-01") \%>\% set_label("Some nonsense date")
crosstable(mtcars2, my_date, by=vs, date_format="\%d/\%m/\%Y")

#Survival data (using formula syntax)
library(survival)
crosstable(aml, Surv(time, status) ~ x, times=c(0,15,30,150), followup=TRUE)

#Patterns
crosstable(mtcars2, vs, by=am, percent_digits=0,
           percent_pattern="{n} ({p_col} / {p_row})")
crosstable(mtcars2, vs, by=am, percent_digits=0,
           percent_pattern="N={n} \np[95\%CI] = {p_col} [{p_col_inf}; {p_col_sup}]")
str_high="n>5"; str_lo="n<=5"
crosstable(mtcars2, vs, by=am, percent_digits=0,
           percent_pattern="col={p_col}, row={p_row} ({ifelse(n<5, str_lo, str_high)})")
}
\seealso{
https://danchaltiel.github.io/crosstable/, as_flextable, import_labels
}
\author{
Dan Chaltiel
}
