% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/queue-abstract.R
\docType{data}
\name{AbstractQueue}
\alias{AbstractQueue}
\title{Defines abstract queue class}
\format{An object of class \code{R6ClassGenerator} of length 24.}
\usage{
AbstractQueue
}
\description{
This class is inspired by \url{https://cran.r-project.org/package=txtq}.
The difference is \code{AbstractQueue} introduce an abstract class that can
be extended and can queue not only text messages, but also arbitrary R
objects, including expressions and environments. All the queue types in this
package inherit this class. See \code{\link[dipsaus]{queue}} for
implementations.
}
\section{Abstract Public Methods}{


Methods start with \code{@...} are not thread-safe. Most of them are not
used directly by users. However, you might want to override them if you
inherit this abstract class. Methods marked as "(override)" are not
implemented, meaning you are supposed to implement the details. Methods
marked as "(optional)" usually have default alternatives.

\describe{
\item{\code{initialize(...)} (override)}{
The constructor. Usually three things to do during the process:
1. set \code{get_locker} \code{free_locker} if you don't want to use the
default lockers. 2. set lock file (if using default lockers). 3. call
\code{self$connect(...)}
}
\item{\code{get_locker()}, \code{free_locker()} (optional)}{
Default is \code{NULL} for each methods, and queue uses an internal
\code{private$default_get_locker} and \code{private$default_free_locker}.
These two methods are for customized locker, please
implement these two methods as functions during \code{self$initialization}
\code{get_locker} obtains and lock access (exclusive), and \code{free_locker}
frees the locker. Once implemented, \code{private$exclusive} will take care
the rest. Type: function; parameters: none; return: none
}
\item{\code{@get_head()}, \code{@set_head(v)} (override)}{
Get head so that we know where we are in the queue \code{self$@get_head()}
should return a integer indicating where we are at the queue
\code{self$@set_head(v)} stores that integer. Parameter \code{v} is always
non-negative, this is guaranteed. Users are not supposed to call these
methods directly, use \code{self$head} and \code{self$head<-} instead.
However, if you inherit this class, you are supposed to override the methods.
}
\item{\code{@get_total()}, \code{@set_total(v)} (override)}{
Similar to \code{@get_head} and \code{@set_head}, defines the total items
ever stored in the queue. total-head equals current items in the queue.
}
\item{\code{@inc_total(n=1)} (optional)}{
Increase total, usually this doesn't need to be override, unless you are
using files to store total and want to decrease number of file connections
}
\item{\code{@append_header(msg, ...)} (override)}{
\code{msg} will be vector of strings, separated by "|", containing encoded
headers: `time`, `key`, `hash`, and `message`. to decode what's inside, you
can use \code{self$print_items(stringr::str_split_fixed(msg, '\\|', 4))}.
\strong{Make sure} to return a number, indicating number of items stored.
Unless handled elsewhere, usually \code{return(length(msg))}.
}
\item{\code{@store_value(value, key)} (override)}{
Defines how to store value. `key` is unique identifier generated from
time, queue ID, and value. Usually I use it as file name or key ID in
database. value is an arbitrary R object to store. you need to store value
somewhere and return a string that will be passed as `hash` in
\code{self$restore_value}.
}
\item{\code{restore_value(hash, key, preserve = FALSE)} (override)}{
Method to restore value from given combination of `hash` and `key`.
`hash` is the string returned by \code{@store_value}, and `key` is the same
as key in \code{@store_value}. preserve is a indicator of whether to
preserve the value for future use. If set to \code{FALSE}, then you are
supposed to free up the resource related to the value. (such as free memory
or disk space)
}
\item{\code{@log(n = -1, all = FALSE) (override)}}{
get \code{n} items from what you saved to during \code{@append_header}.
\code{n} less equal than 0 means listing all possible items.
If \code{all=TRUE}, return all items (number of rows should equals to
\code{self$total}), including popped items. If \code{all=FALSE}, only
return items in the queue (number of rows is \code{self$count}). The
returned value should be a \code{n x 4} matrix. Usually I use
\code{stringr::str_split_fixed(..., '\\|', 4)}. Please see all other
types implemented for example.
}
\item{\code{@reset(...)} (override)}{
Reset queue, remove all items and reset head, total to be 0.
}
\item{\code{@clean()} (override)}{
Clean the queue, remove all the popped items.
}
\item{\code{@validate()} (override)}{
Validate the queue. Stop if the queue is broken.
}
\item{\code{@connect(con, ...)} (override)}{
Set up connection. Usually should be called at the end of
\code{self$initialization} to connect to a database, a folder, or an
existing queue you should do checks whether the connection is new or it's
an existing queue.
}
\item{\code{connect(con, ...)} (optional)}{
Thread-safe version. sometimes you need to override this function instead
of \code{@connect}, because \code{private$exclusive} requires \code{lockfile}
to exist and to be locked. If you don't have lockers ready, or need to set
lockers during the connection, override this one.
}
\item{\code{destroy()} (optional)}{
Destroy a queue, free up space and call
\code{delayedAssign('.lockfile', {stop(...)}, assign.env=private)} to raise
error if a destroyed queue is called again later.
}
}
}

\section{Public Methods}{


Usually don't need to override unless you know what you are doing.

\describe{
\item{\code{push(value, message='',...)}}{
Function to push an arbitrary R object to queue. \code{message} is a string
giving notes to the pushed item. Usually message is stored with header,
separated from values. The goal is to describe the value. \code{...} is
passed to \code{@append_header}
}
\item{\code{pop(n = 1, preserve = FALSE)}}{
Pop \code{n} items from the queue. \code{preserve} indicates whether not to
free up the resources, though not always guaranteed.
}
\item{\code{print_item(item)}, \code{print_items(items)}}{
To decode matrix returned by \code{log()}, returning named list or data frame
with four heads: `time`, `key`, `hash`, and `message`.
}
\item{\code{list(n=-1)}}{
List items in the queue, decoded. If \code{n} is less equal than 0, then
list all results. The result is equivalent to
\code{self$print_items(self$log(n))}
}
\item{\code{log(n=-1,all=FALSE)}}{
List items in the queue, encoded. This is used with \code{self$print_items}.
When \code{all=TRUE}, result will list the  records ever pushed to the queue
since the last time queue is cleaned. When \code{all=FALSE}, results will be
items in the queue. \code{n} is the number of items.
}
}
}

\section{Public Active Bindings}{


\describe{
\item{\code{id}}{
Read-only property. Returns unique ID of current queue.
}
\item{\code{lockfile}}{
The lock file.
}
\item{\code{head}}{
Integer, total number of items popped, i.e. inactive items.
}
\item{\code{total}}{
Total number of items ever pushed to the queue since last cleaned, integer.
}
\item{\code{count}}{
Integer, read-only, equals to total - head, number of active items in the
queue
}
}
}

\section{Private Methods or properties}{


\describe{
\item{\code{.id}}{
Don't use directly. Used to store queue ID.
}
\item{\code{.lockfile}}{
Location of lock file.
}
\item{\code{lock}}{
Preserve the file lock.
}
\item{\code{exclusive(expr,...)}}{
Function to make sure the methods are thread-safe
}
\item{\code{default_get_locker()}}{
Default method to lock a queue
}
\item{\code{default_free_locker}}{
Default method to free a queue
}
}
}

\seealso{
\code{\link[dipsaus]{queue}}
}
\keyword{datasets}
