% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tsl_aggregate.R
\name{tsl_aggregate}
\alias{tsl_aggregate}
\title{Aggregate Time Series List Over Time Periods}
\usage{
tsl_aggregate(tsl = NULL, new_time = NULL, f = mean, ...)
}
\arguments{
\item{tsl}{(required, list) Time series list. Default: NULL}

\item{new_time}{(required, numeric, numeric vector, Date vector, POSIXct vector, or keyword) Definition of the aggregation pattern. The available options are:
\itemize{
\item numeric vector: only for the "numeric" time class, defines the breakpoints for time series aggregation.
\item "Date" or "POSIXct" vector: as above, but for the time classes "Date" and "POSIXct." In any case, the input vector is coerced to the time class of the \code{tsl} argument.
\item numeric: defines fixed time intervals in the units of \code{tsl} for time series aggregation. Used as is when the time class is "numeric", and coerced to integer and interpreted as days for the time classes "Date" and "POSIXct".
\item keyword (see \code{\link[=utils_time_units]{utils_time_units()}}): the common options for the time classes "Date" and "POSIXct" are: "millennia", "centuries", "decades", "years", "quarters", "months", and "weeks". Exclusive keywords for the "POSIXct" time class are: "days", "hours", "minutes", and "seconds". The time class "numeric" accepts keywords coded as scientific numbers, from "1e8" to "1e-8".
}}

\item{f}{(required, function name) Name of function taking a vector as input and returning a single value as output. Typical examples are \code{mean}, \code{max},\code{min}, \code{median}, and \code{quantile}. Default: \code{mean}.}

\item{...}{(optional) further arguments for \code{f}.}
}
\value{
time series list
}
\description{
Time series aggregation involves grouping observations and summarizing group values with a statistical function. This operation is useful to:
\itemize{
\item Decrease (downsampling) the temporal resolution of a time series.
\item Highlight particular states of a time series over time. For example, a daily temperature series can be aggregated by month using \code{max} to represent the highest temperatures each month.
\item Transform irregular time series into regular.
}

This function aggregates time series lists \strong{with overlapping times}. Please check such overlap by assessing the columns "begin" and "end " of the data frame resulting from \code{df <- tsl_time(tsl = tsl)}. Aggregation will be limited by the shortest time series in your time series list. To aggregate non-overlapping time series, please subset the individual components of \code{tsl} one by one either using \code{\link[=tsl_subset]{tsl_subset()}} or the syntax \code{tsl = my_tsl[[i]]}.

\strong{Methods}

Any function returning a single number from a numeric vector can be used to aggregate a time series list. Quoted and unquoted function names can be used. Additional arguments to these functions can be passed via the argument \code{...}. Typical examples are:

\itemize{
\item \code{mean} or \code{"mean"}: see \code{\link[=mean]{mean()}}.
\item \code{median} or \code{"median"}: see \code{\link[stats:median]{stats::median()}}.
\item \code{quantile} or "quantile": see \code{\link[stats:quantile]{stats::quantile()}}.
\item \code{min} or \code{"min"}: see \code{\link[=min]{min()}}.
\item \code{max} or \code{"max"}: see \code{\link[=max]{max()}}.
\item \code{sd} or \code{"sd"}: to compute standard deviation, see \code{\link[stats:sd]{stats::sd()}}.
\item \code{var} or \code{"var"}: to compute the group variance, see \code{\link[stats:cor]{stats::var()}}.
\item \code{length} or \code{"length"}: to compute group length.
\item \code{sum} or \code{"sum"}: see \code{\link[=sum]{sum()}}.
}

This function supports a parallelization setup via \code{\link[future:plan]{future::plan()}}, and progress bars provided by the package \href{https://CRAN.R-project.org/package=progressr}{progressr}.
}
\examples{

# yearly aggregation
#----------------------------------
#long-term monthly temperature of 20 cities
tsl <- tsl_initialize(
  x = cities_temperature,
  name_column = "name",
  time_column = "time"
)

#plot time series
if(interactive()){
  tsl_plot(
    tsl = tsl[1:4],
    guide_columns = 4
  )
}

#check time features
tsl_time(tsl)[, c("name", "resolution", "units")]

#aggregation: mean yearly values
tsl_year <- tsl_aggregate(
  tsl = tsl,
  new_time = "year",
  f = mean
)

#' #check time features
tsl_time(tsl_year)[, c("name", "resolution", "units")]

if(interactive()){
  tsl_plot(
    tsl = tsl_year[1:4],
    guide_columns = 4
  )
}


# other supported keywords
#----------------------------------

#simulate full range of calendar dates
tsl <- tsl_simulate(
  n = 2,
  rows = 1000,
  time_range = c(
    "0000-01-01",
    as.character(Sys.Date())
  )
)

#mean value by millennia (extreme case!!!)
tsl_millennia <- tsl_aggregate(
  tsl = tsl,
  new_time = "millennia",
  f = mean
)

if(interactive()){
  tsl_plot(tsl_millennia)
}

#max value by centuries
tsl_century <- tsl_aggregate(
  tsl = tsl,
  new_time = "century",
  f = max
)

if(interactive()){
  tsl_plot(tsl_century)
}

#quantile 0.75 value by centuries
tsl_centuries <- tsl_aggregate(
  tsl = tsl,
  new_time = "centuries",
  f = stats::quantile,
  probs = 0.75 #argument of stats::quantile()
)

}
\seealso{
\code{\link[=zoo_aggregate]{zoo_aggregate()}}

Other tsl_processing: 
\code{\link{tsl_resample}()},
\code{\link{tsl_smooth}()},
\code{\link{tsl_stats}()},
\code{\link{tsl_transform}()}
}
\concept{tsl_processing}
