% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/zoo_resample.R
\name{zoo_resample}
\alias{zoo_resample}
\title{Resample Zoo Objects to a New Time}
\usage{
zoo_resample(
  x = NULL,
  new_time = NULL,
  method = "linear",
  max_complexity = FALSE
)
}
\arguments{
\item{x}{(required, zoo object) Time series to resample. Default: NULL}

\item{new_time}{(optional, zoo object, keyword, or time vector) New time to resample \code{x} to. The available options are:
\itemize{
\item NULL: a regular version of the time in \code{x} is generated and used for resampling.
\item zoo object: the index of the given zoo object is used as template to resample \code{x}.
\item time vector: a vector with new times to resample \code{x} to. If time in \code{x} is of class "numeric", this vector must be numeric as well. Otherwise, vectors of classes "Date" and "POSIXct" can be used indistinctly.
\item keyword: a valid keyword returned by \code{zoo_time(x)$keywords}, used to generate a time vector with the relevant units.
\item numeric of length 1: interpreted as new time interval, in the highest resolution units returned by \code{zoo_time(x)$units}.
}}

\item{method}{(optional, character string) Name of the method to resample the time series. One of "linear", "spline" or "loess". Default: "linear".}

\item{max_complexity}{(required, logical). Only relevant for methods "spline" and "loess". If TRUE, model optimization is ignored, and the a model of maximum complexity (an overfitted model) is used for resampling. Default: FALSE}
}
\value{
zoo object
}
\description{
\strong{Objective}

Time series resampling involves interpolating new values for time steps not available in the original time series. This operation is useful to:
\itemize{
\item Transform irregular time series into regular.
\item Align time series with different temporal resolutions.
\item Increase (upsampling) or decrease (downsampling) the temporal resolution of a time series.
}

On the other hand, time series resampling \strong{should not be used} to extrapolate new values outside of the original time range of the time series, or to increase the resolution of a time series by a factor of two or more. These operations are known to produce non-sensical results.

\strong{Methods}
This function offers three methods for time series interpolation:

\itemize{
\item "linear" (default): interpolation via piecewise linear regression as implemented in \code{\link[zoo:na.approx]{zoo::na.approx()}}.
\item "spline": cubic smoothing spline regression as implemented in \code{\link[stats:smooth.spline]{stats::smooth.spline()}}.
\item "loess": local polynomial regression fitting as implemented in \code{\link[stats:loess]{stats::loess()}}.
}

These methods are used to fit models \code{y ~ x} where \code{y} represents the values of a univariate time series and \code{x} represents a numeric version of its time.

The functions \code{\link[=utils_optimize_spline]{utils_optimize_spline()}} and \code{\link[=utils_optimize_loess]{utils_optimize_loess()}} are used under the hood to optimize the complexity of the methods "spline" and "loess" by finding the configuration that minimizes the root mean squared error (RMSE) between  observed and predicted \code{y}. However, when the argument \code{max_complexity = TRUE}, the complexity optimization is ignored, and a maximum complexity model is used instead.

\strong{New time}

The argument \code{new_time} offers several alternatives to help define the new time of the resulting time series:

\itemize{
\item \code{NULL}: the target time series (\code{x}) is resampled to a regular time within its original time range and number of observations.
\item \verb{zoo object}: a zoo object to be used as template for resampling. Useful when the objective is equalizing the frequency of two separate zoo objects.
\item \verb{time vector}: a time vector of a class compatible with the time in \code{x}.
\item \code{keyword}: character string defining a resampling keyword, obtained via \code{zoo_time(x, keywords = "resample")$keywords}..
\item \code{numeric}: a single number representing the desired interval between consecutive samples in the units of \code{x} (relevant units can be obtained via \code{zoo_time(x)$units}).
}

\strong{Step by Step}

The steps to resample a time series list are:

\enumerate{
\item The time interpolation range taken from the index of the zoo object. This step ensures that no extrapolation occurs during resampling.
\item If \code{new_time} is provided, any values of \code{new_time} outside of the minimum and maximum interpolation times are removed to avoid extrapolation. If \code{new_time} is not provided, a regular time within the interpolation time range of the zoo object is generated.
\item For each univariate time time series, a model \code{y ~ x}, where \code{y} is the time series and \code{x} is its own time coerced to numeric is fitted.
\itemize{
\item If \code{max_complexity == FALSE} and \code{method = "spline"} or \code{method = "loess"}, the model with the complexity that minimizes the root mean squared error between the observed and predicted \code{y} is returned.
\item If \code{max_complexity == TRUE} and \code{method = "spline"} or \code{method = "loess"}, the first valid model closest to a maximum complexity is returned.
}
\item The fitted model is predicted over \code{new_time} to generate the resampled time series.
}

\strong{Other Details}

Please use this operation with care, as there are limits to the amount of resampling that can be done without distorting the data. The safest option is to keep the distance between new time points within the same magnitude of the distance between the old time points.
}
\examples{
#simulate irregular time series
x <- zoo_simulate(
  cols = 2,
  rows = 50,
  time_range = c("2010-01-01", "2020-01-01"),
  irregular = TRUE
  )

#plot time series
if(interactive()){
  zoo_plot(x)
}

#intervals between samples
x_intervals <- diff(zoo::index(x))
x_intervals

#create regular time from the minimum of the observed intervals
new_time <- seq.Date(
  from = min(zoo::index(x)),
  to = max(zoo::index(x)),
  by = floor(min(x_intervals))
)

new_time
diff(new_time)

#resample using piecewise linear regression
x_linear <- zoo_resample(
  x = x,
  new_time = new_time,
  method = "linear"
)

#resample using max complexity splines
x_spline <- zoo_resample(
  x = x,
  new_time = new_time,
  method = "spline",
  max_complexity = TRUE
)

#resample using max complexity loess
x_loess <- zoo_resample(
  x = x,
  new_time = new_time,
  method = "loess",
  max_complexity = TRUE
)


#intervals between new samples
diff(zoo::index(x_linear))
diff(zoo::index(x_spline))
diff(zoo::index(x_loess))

#plotting results
if(interactive()){

  par(mfrow = c(4, 1), mar = c(3,3,2,2))

  zoo_plot(
    x,
    guide = FALSE,
    title = "Original"
    )

  zoo_plot(
    x_linear,
    guide = FALSE,
    title = "Method: linear"
  )

  zoo_plot(
    x_spline,
    guide = FALSE,
    title = "Method: spline"
    )

  zoo_plot(
    x_loess,
    guide = FALSE,
    title = "Method: loess"
  )

}

}
\seealso{
Other zoo_functions: 
\code{\link{zoo_aggregate}()},
\code{\link{zoo_name_clean}()},
\code{\link{zoo_name_get}()},
\code{\link{zoo_name_set}()},
\code{\link{zoo_permute}()},
\code{\link{zoo_plot}()},
\code{\link{zoo_smooth_exponential}()},
\code{\link{zoo_smooth_window}()},
\code{\link{zoo_time}()},
\code{\link{zoo_to_tsl}()},
\code{\link{zoo_vector_to_matrix}()}
}
\concept{zoo_functions}
