% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/episode_group.R
\name{episode_group}
\alias{episode_group}
\alias{fixed_episodes}
\alias{rolling_episodes}
\title{Episode grouping for case definitions and record deduplication}
\usage{
episode_group(df, sn = NULL, strata = NULL, date, case_length,
  episode_type = "fixed", episode_unit = "days", episodes_max = Inf,
  recurrence_length = NULL, rolls_max = Inf,
  skip_if_b4_lengths = TRUE, data_source = NULL, data_links = "ANY",
  custom_sort = NULL, skip_order = NULL, from_last = FALSE,
  overlap_method = c("exact", "across", "inbetween", "aligns_start",
  "aligns_end", "chain"), overlap_methods = NULL, bi_direction = FALSE,
  group_stats = FALSE, display = TRUE, deduplicate = FALSE,
  to_s4 = TRUE, recurrence_from_last = TRUE,
  case_for_recurrence = FALSE, include_index_period = TRUE)

fixed_episodes(date, sn = NULL, strata = NULL, case_length,
  episode_unit = "days", episodes_max = Inf,
  skip_if_b4_lengths = TRUE, data_source = NULL, data_links = "ANY",
  custom_sort = NULL, skip_order = NULL, from_last = FALSE,
  overlap_method = c("exact", "across", "inbetween", "aligns_start",
  "aligns_end", "chain", "overlap", "none"), overlap_methods = "overlap",
  bi_direction = FALSE, group_stats = FALSE, display = TRUE,
  deduplicate = FALSE, x, to_s4 = TRUE, include_index_period = TRUE)

rolling_episodes(date, sn = NULL, strata = NULL, case_length,
  recurrence_length = NULL, episode_unit = "days",
  episodes_max = Inf, rolls_max = Inf, skip_if_b4_lengths = TRUE,
  data_source = NULL, data_links = "ANY", custom_sort = NULL,
  skip_order = NULL, from_last = FALSE, overlap_method = c("exact",
  "across", "inbetween", "aligns_start", "aligns_end", "chain", "overlap",
  "none"), overlap_methods = "overlap", bi_direction = FALSE,
  group_stats = FALSE, display = TRUE, deduplicate = FALSE, x,
  to_s4 = TRUE, recurrence_from_last = TRUE,
  case_for_recurrence = FALSE, include_index_period = TRUE)
}
\arguments{
\item{df}{\code{data.frame}. One or more datasets appended together.}

\item{sn}{Unique numerical record identifier. Optional.}

\item{strata}{Subsets of the dataset. Episode grouping will be done separately within each subset of the dataset.
In \code{\link{episode_group}}, you can use multiple columns. \code{\link{record_group}} can create useful \code{strata} e.g. patient identifiers.}

\item{date}{Event date (\code{date}, \code{datetime} or \code{numeric}) or period (\code{\link{number_line}}).}

\item{case_length}{Duration after a \code{"case"} within which subsequent events are considered \code{"duplicate"} events.
This period is referred to as the the \code{case window}. Can be a (\code{\link{number_line}}) range.}

\item{episode_type}{\code{"fixed"} or \code{"rolling"}.}

\item{episode_unit}{Time units. Options are "seconds", "minutes", "hours", "days", "weeks", "months" or "years". See \code{diyar::episode_unit}.}

\item{episodes_max}{Maximum number of episodes to have within each \code{strata}.}

\item{recurrence_length}{Duration after the last or first event (see \code{recurrence_from_last}) of the previous window within which subsequent events are considered \code{"recurrent"} events.
This period is referred to as the \code{recurrence window}. If \code{recurrence_length} is not supplied, it's assumed to be the same as \code{case_length}.
Can be a (\code{\link{number_line}}) range.}

\item{rolls_max}{Maximum number of times an episode can reoccur. Only used if \code{episode_type} is \code{"rolling"}.}

\item{skip_if_b4_lengths}{If \code{TRUE} (default), records \code{events} before the \code{case_length} or the \code{recurrence_length} range are skipped.}

\item{data_source}{Unique dataset identifier. Useful when the dataset contains data from multiple sources.
In \code{\link{episode_group}}, you can use multiple columns supplied as column names.}

\item{data_links}{Breakup episodes that will not include records from these \code{data_sources}. \code{data_links} should be a \code{list} with every element named 'l' (links) or 'g' (groups).
Useful in skipping episodes that are not required to minimise processing time. Ignored if \code{data_source} is \code{NULL}.}

\item{custom_sort}{Preferential order for \code{"case"} assignment. Useful in specifying that episode grouping begins at particular events regardless of chronological order.
In \code{\link{episode_group}}, you can use multiple columns as sort levels.}

\item{skip_order}{Skip episodes whose \code{case} events have \code{custom_sort} values that are less than or equal to the \code{"nth"} level of \code{custom_sort}.
Useful in skipping episodes that are not required and so minimises the overall processing time. Ignored if \code{custom_sort} is \code{NULL}.}

\item{from_last}{If \code{TRUE}, episode grouping will be backwards in time - starting at the most recent event and proceeding to the earliest.
If \code{FALSE}, it'll be forward in time - starting at the earliest event and proceeding to the most recent one.}

\item{overlap_method}{Methods of overlap considered when grouping event periods. Each pair of periods are checked with the same set of \code{overlap_method}.
Deprecated please use \code{overlap_methods} instead.}

\item{overlap_methods}{Methods of overlap considered when grouping event periods. Different pairs of periods can be checked with different sets of \code{overlap_methods}}

\item{bi_direction}{If \code{FALSE} (default), \code{"duplicate"} events will be those within the \code{case_length} before \strong{or} after the \code{"case"} as determined by \code{from_last}.
If \code{TRUE}, \code{"duplicate"} events will be those within the same period before \strong{and} after the \code{"case"}.}

\item{group_stats}{If \code{TRUE}, the output will include additional information with useful stats for each episode group.}

\item{display}{If \code{TRUE} (default), a progress message is printed on screen.}

\item{deduplicate}{if \code{TRUE}, \code{"dupilcate"} events are excluded from the output.}

\item{to_s4}{If \code{TRUE} (default), episodes are returned as an \code{\link[=epid-class]{epid}} object.}

\item{recurrence_from_last}{If \code{TRUE} (default), the reference event for a \code{recurrence window} will be the last event from the previous window.
If \code{FALSE} (default), it will be the first event. Only used if \code{episode_type} is \code{"rolling"}.}

\item{case_for_recurrence}{If \code{TRUE}, both case and recurrence events will have a \code{case window}.
If \code{FALSE} (default), only \code{case events} will have a \code{case window}. Only used if \code{episode_type} is \code{"rolling"}.}

\item{include_index_period}{If \code{TRUE}, overlaps with the index event or period are grouped together even if they are outside the cut-off range (\code{case_length} or \code{recurrence_length}).}

\item{x}{Record date or interval. Deprecated. Please use \code{date}}
}
\value{


\code{\link[=epid-class]{epid}} objects or \code{data.frame} if \code{to_s4} is \code{FALSE})

\itemize{
\item \code{sn} - unique record identifier as provided (or generated)
\item \code{epid | .Data} - unique episode identifier
\item \code{wind_id} - unique window identifier
\item \code{wind_nm} - type of window i.e. "Case" or "Recurrence"
\item \code{case_nm} - record type in regards to case assignment
\item \code{dist_from_wind} - duration of each event from its window's reference event
\item \code{dist_from_epid} - duration of each event from its episode's reference event
\item \code{epid_total} - number of records in each episode
\item \code{epid_dataset} - data sources in each episode
\item \code{epid_interval} - episode start and end dates. A \code{\link{number_line}} object.
\item \code{epid_length} - difference between episode start and end dates (\code{difftime}). If possible, it's the same unit as \code{episode_unit} otherwise, a difference in days is returned
\item \code{epid_total} - number of records in each episode

}
}
\description{
Group events into chronological episodes
}
\details{
Episode grouping begins at a reference event (\code{"case"}) and proceeds forward or backward in time depending on \code{from_last}.
If \code{custom_sort} is used, episode grouping can be forced to begin at certain events before proceeding forward or backwards in time.
The maximum duration of a \code{"fixed"} episode is the \code{case_length}. This period is referred to as the \code{case window}. The maximum duration of a \code{"rolling"} episode is the
\code{case_length} plus all periods of recurrence. The recurrence periods are referred to as \code{recurrence windows}.
This is a specified duration (\code{recurrence_length}) after the last or first (depending on \code{recurrence_from_last}) event in the previous window.
Events within this period are considered \code{"recurrent"} events.

When a \code{data_source} identifier is provided,
\code{epid_dataset} is included in the output. This lists the source of every event in each episode.

\code{fixed_episodes()} and \code{rolling_episodes()} are wrapper functions of \code{episode_group()}.
They are convenient alternatives with the same functionalities.
}
\examples{
library(diyar)
data(infections)
data(hospital_admissions)

db_1 <- infections
db_1$patient_id <- c(rep("PID 1",8), rep("PID 2",3))

# Fixed episodes
# One 16-day (15-day difference) episode per patient
db_1$epids_p <- fixed_episodes(date=db_1$date, strata = db_1$patient_id,
case_length = 15, episodes_max = 1, display = FALSE)

# Rolling episodes
# Case length of 16 days and recurrence periods of 11 days
db_1$rd_b <- rolling_episodes(db_1$date, case_length = 15,
recurrence_length = 10, display = FALSE)

# Interval grouping
hospital_admissions$admin_period <- number_line(hospital_admissions$admin_dt,
hospital_admissions$discharge_dt)
admissions <- hospital_admissions[c("admin_period","epi_len")]

# Episodes of overlaping periods of admission
hospital_admissions$epi_0 <- fixed_episodes(date=hospital_admissions$admin_period,
case_length = 0, group_stats = TRUE, to_s4=TRUE)

# Note - episode_group() takes column names not actual values

}
\seealso{
\code{\link{record_group}}, \code{\link{overlaps}} and \code{\link{number_line}}
}
