% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ipfN2marginals.R
\name{ipf2N2}
\alias{ipf2N2}
\title{Iterative proportional fitting of an abundance table to Hill-N2 marginals}
\usage{
ipf2N2(
  Y,
  max_iter = 100,
  updateN2 = TRUE,
  N2N_N2_species = TRUE,
  N2N_N2_sites = FALSE,
  crit = 0.01
)
}
\arguments{
\item{Y}{abundance table (matrix or dataframe-like), ideally, 
with names for rows and columns. 
BEWARE: all rows and columns should have positive sums!}

\item{max_iter}{maximum number of iterative proportional fitting (ipf) 
iterations.}

\item{updateN2}{logical, default \code{TRUE}.
If \code{FALSE} the marginal sums are proportional to 
the N2-marginals of the initial table, but the N2-marginals of the returned 
matrix may not be equal to their marginal sum.
If \code{updateN2 = TRUE} and \code{N2N_N2_species=TRUE} (the default), 
the column marginals are \code{N2(N-N2)/2} with \code{N} the number of sites. 
The row sums are then proportional to, what we term, the effective number of
informative species.
If \code{N2N_N2_species = FALSE},                                
the returned transformed table has N2 columns marginals, 
\emph{i.e.} \code{colSums(Y2) = const*N2species(Y2)} with \code{Y2} 
the return value of \code{ipf2N2} and \code{const} a constant.
If converged, N2 row marginals are equal to the row sums, \emph{i.e.} 
\code{rowSums(Y2) = approx. N2sites(Y2)}.}

\item{N2N_N2_species}{Set marginals proportional to \code{N2(N-N2)} Default 
\code{TRUE}.}

\item{N2N_N2_sites}{Default \code{FALSE}. Do not change.}

\item{crit}{stopping criterion.}
}
\value{
a matrix of the same order as the input \code{Y},
obtained after ipf to N2-marginals.
}
\description{
Function for preprocessing/transforming an abundance table
by iterative proportional fitting,  
so that the transformed table has marginals \code{N2} or \code{N2(N-N2)}
with \code{N} the number of elements in the margin.
Hill-N2 is the effective number of species. It is of intrinsic interest in
weighted averaging (CWM and SNC) as their variance is approximately 
inversely proportional to N2 (ter Braak 2019), 
and therefore of interest in \code{\link{dc_CA}}.
}
\details{
Applying \code{ipf2N2} with \code{N2N_N2_species=FALSE} 
to an presence-absence data table returns the same table.
However, a species that occurs everywhere (or in most of the sites)
is not very informative. This is acknowledged with the default option
\code{N2N_N2_species=TRUE}. Then,
with \code{N2N_N2_species=TRUE}, species that occur
in more than halve the number of sites are down-weighted, so that
the row sum is no longer equal to the richness of the site (the number of species),
but proportional to the number of informative species.
}
\examples{
data("dune_trait_env")

# rownames are carried forward in results
rownames(dune_trait_env$comm) <- dune_trait_env$comm$Sites
Y <- dune_trait_env$comm[, -1] # must delete "Sites"
Y_N2 <- ipf2N2(Y, updateN2 = FALSE, N2N_N2_species = FALSE)
attr(Y_N2, "iter") # 4

# show that column margins of the transform matrix are
# equal to the Hill N2 values
diff(range(colSums(Y_N2) / apply(X = Y, MARGIN = 2, FUN = fN2))) #  8.881784e-16
diff(range(rowSums(Y_N2) / apply(X = Y, MARGIN = 1, FUN = fN2))) #  0.07077207

Y_N2i <- ipf2N2(Y, updateN2 = TRUE, N2N_N2_species = FALSE)
attr(Y_N2i, "iter") # 5
diff(range(colSums(Y_N2i) / apply(X = Y_N2i, MARGIN = 2, FUN = fN2))) # 2.220446e-15
diff(range(rowSums(Y_N2i) / apply(X = Y_N2i, MARGIN = 1, FUN = fN2))) # 0.105742

# the default version:
Y_N2N_N2i <- ipf2N2(Y)
# ie. 
# Y_N2N_N2i <- ipf2N2(Y, updateN2 = TRUE, N2N_N2_species = TRUE)
attr(Y_N2N_N2i, "iter") # 16
N2 <- apply(X = Y_N2N_N2i, MARGIN = 2, FUN = fN2)
N <- nrow(Y)
diff(range(colSums(Y_N2N_N2i) / (N2 * (N - N2)))) # 2.220446e-16

N2_sites <- apply(X = Y_N2N_N2i, MARGIN = 1, FUN = fN2)
R <- rowSums(Y_N2N_N2i)
N * max(N2_sites / sum(N2_sites) - R / sum(R)) # 0.009579165

sum(Y_N2N_N2i) - sum(Y)

mod0 <- dc_CA(formulaEnv = ~ A1 + Moist + Mag + Use + Manure,
              formulaTraits = ~ SLA + Height + LDMC + Seedmass + Lifespan,
              response = Y,  
              dataEnv = dune_trait_env$envir,
              dataTraits = dune_trait_env$traits, 
              divide = FALSE,
              verbose = FALSE)

mod1 <- dc_CA(formulaEnv = ~ A1 + Moist + Mag + Use + Manure,
              formulaTraits = ~ SLA + Height + LDMC + Seedmass + Lifespan,
              response = Y_N2N_N2i,  
              dataEnv = dune_trait_env$envir,
              dataTraits = dune_trait_env$traits, 
              verbose = FALSE)

mod1$eigenvalues / mod0$eigenvalues
# ratios of eigenvalues greater than 1,
# indicate axes with higher (squared) fourth-corner correlation

# ipf2N2 for a presence-absence data matrix																   
Y_PA <- 1 * (Y > 0)
Y_PA_N2 <- ipf2N2(Y_PA, N2N_N2_species = FALSE)
attr(Y_PA_N2, "iter") # 1
diff(range(Y_PA - Y_PA_N2)) # 4.440892e-16, i.e no change

Y_PA_N2i <- ipf2N2(Y_PA, N2N_N2_species = TRUE)
attr(Y_PA_N2i, "iter") # 9
N_occ <- colSums(Y_PA) # number of occurrences of species
N <- nrow(Y_PA)
plot(N_occ, colSums(Y_PA_N2i))
cor(colSums(Y_PA_N2i), N_occ * (N - N_occ)) # 0.9916
mod2 <- dc_CA(formulaEnv = ~ A1 + Moist + Mag + Use + Manure,
              formulaTraits = ~ SLA + Height + LDMC + Seedmass + Lifespan,
              response = Y_PA,  
              dataEnv = dune_trait_env$envir,
              dataTraits = dune_trait_env$traits,
              divideBySiteTotals = FALSE,
              verbose = FALSE)
			  
mod3 <- dc_CA(formulaEnv = ~ A1 + Moist + Mag + Use + Manure,
              formulaTraits = ~ SLA + Height + LDMC + Seedmass + Lifespan,
              response = Y_PA_N2i,  
              dataEnv = dune_trait_env$envir,
              dataTraits = dune_trait_env$traits,
              verbose = FALSE)
			  
mod3$eigenvalues / mod2$eigenvalues
# ratios of eigenvalues greater than 1,
# indicate axes with higher (squared) fourth-corner correlation
}
\references{
ter Braak, C.J.F. (2019). 
New robust weighted averaging- and model-based methods
for assessing trait-environment relationships. 
Methods in Ecology and Evolution, 10 (11), 1962-1971. 
\doi{10.1111/2041-210X.13278}
}
