% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/colwise-mutate.R
\name{mutate_all}
\alias{mutate_all}
\alias{mutate_if}
\alias{mutate_at}
\alias{transmute_all}
\alias{transmute_if}
\alias{transmute_at}
\title{Mutate multiple columns}
\usage{
mutate_all(.tbl, .funs, ...)

mutate_if(.tbl, .predicate, .funs, ...)

mutate_at(.tbl, .vars, .funs, ..., .cols = NULL)

transmute_all(.tbl, .funs, ...)

transmute_if(.tbl, .predicate, .funs, ...)

transmute_at(.tbl, .vars, .funs, ..., .cols = NULL)
}
\arguments{
\item{.tbl}{A \code{tbl} object.}

\item{.funs}{A function \code{fun}, a quosure style lambda \code{~ fun(.)} or a list of either form.}

\item{...}{Additional arguments for the function calls in
\code{.funs}. These are evaluated only once, with \link[rlang:dyn-dots]{tidy dots} support.}

\item{.predicate}{A predicate function to be applied to the columns
or a logical vector. The variables for which \code{.predicate} is or
returns \code{TRUE} are selected. This argument is passed to
\code{\link[rlang:as_function]{rlang::as_function()}} and thus supports quosure-style lambda
functions and strings representing function names.}

\item{.vars}{A list of columns generated by \code{\link[=vars]{vars()}},
a character vector of column names, a numeric vector of column
positions, or \code{NULL}.}

\item{.cols}{This argument has been renamed to \code{.vars} to fit
dplyr's terminology and is deprecated.}
}
\value{
A data frame. By default, the newly created columns have the shortest
names needed to uniquely identify the output. To force inclusion of a name,
even when not needed, name the input (see examples for details).
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#superseded}{\figure{lifecycle-superseded.svg}{options: alt='[Superseded]'}}}{\strong{[Superseded]}}

Scoped verbs (\verb{_if}, \verb{_at}, \verb{_all}) have been superseded by the use of
\code{\link[=pick]{pick()}} or \code{\link[=across]{across()}} in an existing verb. See \code{vignette("colwise")} for
details.

The \link{scoped} variants of \code{\link[=mutate]{mutate()}} and \code{\link[=transmute]{transmute()}} make it easy to apply
the same transformation to multiple variables. There are three variants:
\itemize{
\item _all affects every variable
\item _at affects variables selected with a character vector or vars()
\item _if affects variables selected with a predicate function:
}
}
\section{Grouping variables}{


If applied on a grouped tibble, these operations are \emph{not} applied
to the grouping variables. The behaviour depends on whether the
selection is \strong{implicit} (\code{all} and \code{if} selections) or
\strong{explicit} (\code{at} selections).
\itemize{
\item Grouping variables covered by explicit selections in
\code{mutate_at()} and \code{transmute_at()} are always an error. Add
\code{-group_cols()} to the \code{\link[=vars]{vars()}} selection to avoid this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{data \%>\% mutate_at(vars(-group_cols(), ...), myoperation)
}\if{html}{\out{</div>}}

Or remove \code{group_vars()} from the character vector of column names:

\if{html}{\out{<div class="sourceCode">}}\preformatted{nms <- setdiff(nms, group_vars(data))
data \%>\% mutate_at(vars, myoperation)
}\if{html}{\out{</div>}}
\item Grouping variables covered by implicit selections are ignored by
\code{mutate_all()}, \code{transmute_all()}, \code{mutate_if()}, and
\code{transmute_if()}.
}
}

\section{Naming}{


The names of the new columns are derived from the names of the
input variables and the names of the functions.
\itemize{
\item if there is only one unnamed function (i.e. if \code{.funs} is an unnamed list
of length one),
the names of the input variables are used to name the new columns;
\item for \verb{_at} functions, if there is only one unnamed variable (i.e.,
if \code{.vars} is of the form \code{vars(a_single_column)}) and \code{.funs} has length
greater than one,
the names of the functions are used to name the new columns;
\item otherwise, the new names are created by
concatenating the names of the input variables and the names of the
functions, separated with an underscore \code{"_"}.
}

The \code{.funs} argument can be a named or unnamed list.
If a function is unnamed and the name cannot be derived automatically,
a name of the form "fn#" is used.
Similarly, \code{\link[=vars]{vars()}} accepts named and unnamed arguments.
If a variable in \code{.vars} is named, a new column by that name will be created.

Name collisions in the new columns are disambiguated using a unique suffix.
}

\examples{
iris <- as_tibble(iris)

# All variants can be passed functions and additional arguments,
# purrr-style. The _at() variants directly support strings. Here
# we'll scale the variables `height` and `mass`:
scale2 <- function(x, na.rm = FALSE) (x - mean(x, na.rm = na.rm)) / sd(x, na.rm)
starwars \%>\% mutate_at(c("height", "mass"), scale2)
# ->
starwars \%>\% mutate(across(c("height", "mass"), scale2))

# You can pass additional arguments to the function:
starwars \%>\% mutate_at(c("height", "mass"), scale2, na.rm = TRUE)
starwars \%>\% mutate_at(c("height", "mass"), ~scale2(., na.rm = TRUE))
# ->
starwars \%>\% mutate(across(c("height", "mass"), ~ scale2(.x, na.rm = TRUE)))

# You can also supply selection helpers to _at() functions but you have
# to quote them with vars():
iris \%>\% mutate_at(vars(matches("Sepal")), log)
iris \%>\% mutate(across(matches("Sepal"), log))

# The _if() variants apply a predicate function (a function that
# returns TRUE or FALSE) to determine the relevant subset of
# columns. Here we divide all the numeric columns by 100:
starwars \%>\% mutate_if(is.numeric, scale2, na.rm = TRUE)
starwars \%>\% mutate(across(where(is.numeric), ~ scale2(.x, na.rm = TRUE)))

# mutate_if() is particularly useful for transforming variables from
# one type to another
iris \%>\% mutate_if(is.factor, as.character)
iris \%>\% mutate_if(is.double, as.integer)
# ->
iris \%>\% mutate(across(where(is.factor), as.character))
iris \%>\% mutate(across(where(is.double), as.integer))

# Multiple transformations ----------------------------------------

# If you want to apply multiple transformations, pass a list of
# functions. When there are multiple functions, they create new
# variables instead of modifying the variables in place:
iris \%>\% mutate_if(is.numeric, list(scale2, log))
iris \%>\% mutate_if(is.numeric, list(~scale2(.), ~log(.)))
iris \%>\% mutate_if(is.numeric, list(scale = scale2, log = log))
# ->
iris \%>\%
  as_tibble() \%>\%
  mutate(across(where(is.numeric), list(scale = scale2, log = log)))

# When there's only one function in the list, it modifies existing
# variables in place. Give it a name to instead create new variables:
iris \%>\% mutate_if(is.numeric, list(scale2))
iris \%>\% mutate_if(is.numeric, list(scale = scale2))
}
\seealso{
\link[=scoped]{The other scoped verbs}, \code{\link[=vars]{vars()}}
}
\keyword{internal}
