% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/computeTargetValues.R
\name{computeTargetValues}
\alias{computeTargetValues}
\alias{computeTargetValues.default}
\alias{computeTargetValues.factor}
\title{Compute target values based on the item pool.}
\usage{
computeTargetValues(
  itemValues,
  nForms,
  testLength = NULL,
  allowedDeviation = NULL,
  relative = FALSE
)

\method{computeTargetValues}{default}(
  itemValues,
  nForms,
  testLength = NULL,
  allowedDeviation = NULL,
  relative = FALSE
)

\method{computeTargetValues}{factor}(
  itemValues,
  nForms,
  testLength = NULL,
  allowedDeviation = NULL,
  relative = FALSE
)
}
\arguments{
\item{itemValues}{Item parameter/values for which the sum per test form should be constrained.}

\item{nForms}{Number of forms to be created.}

\item{testLength}{to be documented.}

\item{allowedDeviation}{Numeric value of length 1. How much deviance is allowed from target values?}

\item{relative}{Is the \code{allowedDeviation} expressed as a proportion?}
}
\value{
a vector or a matrix with target values (see details)
}
\description{
Compute target values for item values/categories based on the number
of items in the item pool, the number of test forms to assemble and the number
of items in each test form (i.e., test length).
}
\details{
Both for numerical and categorical item values, the target values are the item
pool average scaled by the ratio of items in the forms and items in the item
pool. The behavior of the function changes depending on the class of
\code{itemValues}.

When \code{itemValues} is a numerical vector, an when \code{allowedDeviation}
is \code{NULL} (the default), only one target value is computed. This value
could be used in the \code{targetConstraint}-function. Otherwise (i.e.,
\code{allowedDeviation} is a numerical value), the target is computed, but a
minimal and a maximal (target)value are returned, based on the allowed
deviation. When \code{relative == TRUE} the allowed deviation should be
expressed as a proportion. In that case the minimal and maximal values are
a computed proportionally.

When \code{itemValues} is a \code{factor}, it is assumed that the item values
are item categories, and hence only whole valued frequencies are returned.
To be more precise, a matrix with the minimal and maximal target frequencies
for every level of the factor are returned. When \code{allowedDeviation}
is \code{NULL}, the difference between the minimal and maximal value is
one (or zero). As a consequence, dummy-item values are best specified as a
factor (see examples).
}
\section{Methods (by class)}{
\itemize{
\item \code{default}: compute target values

\item \code{factor}: compute target frequencies for item categories
}}

\examples{
## Assume an item pool with 50 items with random item information values (iif) for
## a given ability value.
set.seed(50)
itemInformations <- runif(50, 0.5, 3)

## The target value for the test information value (i.e., sum of the item
## informations) when three test forms of 10 items are assembled is:
computeTargetValues(itemInformations, nForms = 3, testLength = 10)

## The minimum and maximum test iformation values for an allowed deviation of
## 10 percent are:
computeTargetValues(itemInformations, nForms = 3, allowedDeviation = .10,
   relative = TRUE, testLength = 10)


## items$MC is dummy variable indication which items in the pool are multiple choise
str(items$MC)

## when used as a numerical vector, the dummy is not treated as a categorical
## indicator, but rather as a numerical value.
computeTargetValues(items$MC, nForms = 14)
computeTargetValues(items$MC, nForms = 14, allowedDeviation = 1)

## Therefore, it is best to convert dummy variables into a factor, so that
## automatically freqyencies are returned
MC_factor <- factor(items$MC, labels = c("not MC", "MC"))
computeTargetValues(MC_factor, nForms = 14)
computeTargetValues(MC_factor, nForms = 3)

}
