% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ebproject.R
\name{ebreg}
\alias{ebreg}
\title{Implements the empirical Bayes method in high-dimensional linear model setting for inference and prediction}
\usage{
ebreg(
  y,
  X,
  XX,
  standardized = TRUE,
  alpha = 0.99,
  gam = 0.005,
  sig2,
  prior = TRUE,
  igpar = c(0.01, 4),
  log.f,
  M,
  sample.beta = FALSE,
  pred = FALSE,
  conf.level = 0.95
)
}
\arguments{
\item{y}{vector of response variables for regression}

\item{X}{matrix of predictor variables}

\item{XX}{vector to predict outcome variable, if pred=TRUE}

\item{standardized}{logical. If TRUE, the data provided has already been standardized}

\item{alpha}{numeric value between 0 and 1, likelihood fraction. Default is 0.99}

\item{gam}{numeric value between 0 and 1, conditional prior precision parameter. Default is 0.005}

\item{sig2}{numeric value for error variance. If NULL (default), variance is estimated from data}

\item{prior}{logical. If TRUE, a prior is used for the error variance}

\item{igpar}{the parameters for the inverse gamma prior on the error variance. Default is (0.01,4)}

\item{log.f}{log of the prior for the model size}

\item{M}{integer value to indicate the Monte Carlo sample size (burn-in of size 0.2 * M automatically added)}

\item{sample.beta}{logical. If TRUE, samples of beta are obtained}

\item{pred}{logical. If TRUE, predictions are obtained}

\item{conf.level}{numeric value between 0 and 1, confidence level for the marginal credible interval if sample.beta=TRUE, and for the prediction interval if pred=TRUE}
}
\value{
A list with components
\itemize{
 \item beta - matrix with rows containing sampled beta, if sample.beta=TRUE, otherwise NULL
 \item beta.mean - vector containing the posterior mean of beta, if sample.beta=TRUE, otherwise NULL
 \item ynew - matrix containing predicted responses, if pred=TRUE, otherwise NULL
 \item ynew.mean - vector containing the predictions for the predictor values tested, XX, if pred=TRUE, otherwise NULL
 \item S - matrix with rows containing the sampled models
 \item incl.prob - vector containing inclusion probabilities of the predictors
 \item sig2 - estimated error variance, if prior=FALSE, otherwise NULL
 \item PI - prediction interval, confidence level specified by the user, if pred=TRUE, otherwise NULL
 \item CI - matrix containing marginal credible intervals, confidence level specified by the user, if sample.beta=TRUE, otherwise NULL
}
}
\description{
The function ebreg implements the method first presented in Martin, Mess, and Walker (2017) for
Bayesian inference and variable selection in the high-dimensional sparse linear regression problem.  The
chief novelty is the manner in which the prior distribution for the regression coefficients depends on data;
more details, with a focus on the prediction problem, are given in Martin and Tang (2019).
}
\details{
Consider the classical regression problem
\deqn{y = X\beta + \sigma \epsilon,}{y = X\beta + \sigma \epsilon,}
where \eqn{y} is a \eqn{n}-vector of responses, \eqn{X} is a \eqn{n \times p}{n x p} matrix of predictor variables,
\eqn{\beta} is a \eqn{p}-vector of regression coefficients, \eqn{\sigma > 0} is a scale parameter, and
\eqn{\epsilon} is a \eqn{n}-vector of independent and identically distributed standard normal random errors.
Here we allow \eqn{p \ge n}{p \ge n} (or even \eqn{p \gg n}{p >> n}) and accommodate the high dimensionality by assuming
\eqn{\beta} is sparse in the sense that most of its components are zero.  The approach described in
Martin, Mess, and Walker (2017) and in Martin and Tang (2019) starts by decomposing the full \eqn{\beta}
vector as a pair \eqn{(S, \beta_S)} where \eqn{S} is a subset of indices \eqn{1,2,\ldots,p} that represents the
location of active variables and \eqn{\beta_S} is the \eqn{|S|}-vector of non-zero coefficients.  The approach
proceeds by specifying a prior distribution for \eqn{S} and then a conditional prior distribution for
\eqn{\beta_S}, given \eqn{S}.  This latter prior distribution here is taken to depend on data, hence "empirical".
A prior distribution for \eqn{\sigma^2} can also be introduced, and this option is included in the function.
}
\examples{
n <- 70
p <- 100
beta <- rep(1, 5)
s0 <- length(beta)
sig2 <- 1
d <- 1
log.f <- function(x) -x * (log(1) + 0.05 * log(p)) + log(x <= n)
X <- matrix(rnorm(n * p), nrow=n, ncol=p)
X.new <- matrix(rnorm(p), nrow=1, ncol=p)
y <- as.numeric(X[, 1:s0] \%*\% beta[1:s0]) + sqrt(sig2) * rnorm(n)

o<-ebreg(y, X, X.new, TRUE, .99, .005, NULL, FALSE, igpar=c(0.01, 4),
log.f, M=5000, TRUE, FALSE, .95)

incl.pr <- o$incl.prob
plot(incl.pr, xlab="Variable Index", ylab="Inclusion Probability", type="h", ylim=c(0,1))

}
\references{
\insertRef{martin.mess.walker.eb}{ebreg}

\insertRef{martin2019empirical}{ebreg}
}
\author{
Yiqi Tang

Ryan Martin
}
