% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/biprobit_latent.R
\name{biprobit_latent}
\alias{biprobit_latent}
\title{Recursive Bivariate Probit Model with Latent First Stage}
\usage{
biprobit_latent(
  form1,
  form2,
  data = NULL,
  EM = FALSE,
  par = NULL,
  method = "BFGS",
  verbose = 0,
  maxIter = 500,
  tol = 1e-05,
  tol_LL = 1e-06
)
}
\arguments{
\item{form1}{Formula for the first probit model, in which the dependent variable is unobserved. Use a formula like ~w to avoid specifying the dependent variable.}

\item{form2}{Formula for the second probit model, the latent dependent variable of the first stage is automatically added as a regressor in this model}

\item{data}{Input data, a data frame}

\item{EM}{Whether to maximize likelihood use the Expectation-Maximization (EM) algorithm, which is slower but more robust. Defaults to FLASE, but should change to TRUE is the model has convergence issues.}

\item{par}{Starting values for estimates}

\item{method}{Optimization algorithm. Default is BFGS}

\item{verbose}{A integer indicating how much output to display during the estimation process.
\itemize{
\item <0 - No ouput
\item 0 - Basic output (model estimates)
\item 1 - Moderate output, basic ouput + parameter and likelihood in each iteration
\item 2 - Extensive output, moderate output + gradient values on each call
}}

\item{maxIter}{max iterations for EM algorithm}

\item{tol}{tolerance for convergence of EM algorithm}

\item{tol_LL}{tolerance for convergence of likelihood}
}
\value{
A list containing the results of the estimated model, some of which are inherited from the return of maxLik
\itemize{
\item estimates: Model estimates with 95\% confidence intervals. Prefix "1" means first stage variables.
\item estimate or par: Point estimates
\item variance_type: covariance matrix used to calculate standard errors. Either BHHH or Hessian.
\item var: covariance matrix
\item se: standard errors
\item gradient: Gradient function at maximum
\item hessian: Hessian matrix at maximum
\item gtHg: \eqn{g'H^-1g}, where H^-1 is simply the covariance matrix. A value close to zero (e.g., <1e-3 or 1e-6) indicates good convergence.
\item LL or maximum: Likelihood
\item AIC: AIC
\item BIC: BIC
\item n_obs: Number of observations
\item n_par: Number of parameters
\item iterations: number of iterations taken to converge
\item message: Message regarding convergence status.
}

Note that the list inherits all the components in the output of maxLik. See the documentation of maxLik for more details.
}
\description{
Estimate two probit models with bivariate normally distributed error terms, in which the dependent variable of the first stage model is unobserved.\cr\cr
First stage (Probit, \eqn{m_i^*} is unobserved):
\deqn{m_i^*=1(\boldsymbol{\alpha}'\mathbf{w_i}+u_i>0)}{m_i^* = 1(\alpha' * w_i + u_i > 0)}
Second stage (Probit):
\deqn{y_i = 1(\boldsymbol{\beta}'\mathbf{x_i} + {\gamma}m_i^* + \sigma v_i>0)}{y_i = 1(\beta' * x_i + \gamma * m_i^* + \sigma * v_i > 0)}
Endogeneity structure:
\eqn{u_i} and \eqn{v_i} are bivariate normally distributed with a correlation of \eqn{\rho}. \cr\cr
w and x can be the same set of variables. The identification of this model is generally weak, especially if w are not good predictors of m. \eqn{\gamma} is assumed to be positive to ensure that the model estimates are unique.
}
\examples{
\donttest{
library(MASS)
N = 2000
rho = -0.5
set.seed(1)

x = rbinom(N, 1, 0.5)
z = rnorm(N)

e = mvrnorm(N, mu=c(0,0), Sigma=matrix(c(1,rho,rho,1), nrow=2))
e1 = e[,1]
e2 = e[,2]

m = as.numeric(1 + x + z + e1 > 0)
y = as.numeric(1 + x + z + m + e2 > 0)

est = biprobit(m~x+z, y~x+z+m)
print(est$estimates, digits=3)

est_latent = biprobit_latent(~x+z, y~x+z)
print(est_latent$estimates, digits=3)
}
}
\references{
Peng, Jing. (2023) Identification of Causal Mechanisms from Randomized Experiments: A Framework for Endogenous Mediation Analysis. Information Systems Research, 34(1):67-84. Available at https://doi.org/10.1287/isre.2022.1113
}
\seealso{
Other endogeneity: 
\code{\link{bilinear}()},
\code{\link{biprobit_partial}()},
\code{\link{biprobit}()},
\code{\link{linear_probit}()},
\code{\link{pln_linear}()},
\code{\link{pln_probit}()},
\code{\link{probit_linearRE}()},
\code{\link{probit_linear_latent}()},
\code{\link{probit_linear_partial}()},
\code{\link{probit_linear}()}
}
\concept{endogeneity}
