% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nnevclus.R
\name{nnevclus}
\alias{nnevclus}
\title{NN-EVCLUS algorithm}
\usage{
nnevclus(
  x,
  k = n - 1,
  D = NULL,
  J = NULL,
  c,
  type = "simple",
  n_H,
  ntrials = 1,
  d0 = quantile(D, 0.9),
  fhat = NULL,
  lambda = 0,
  y = NULL,
  Is = NULL,
  nu = 0,
  ML = NULL,
  CL = NULL,
  xi = 0,
  tr = FALSE,
  options = c(1, 1000, 1e-04, 10),
  param0 = list(U0 = NULL, V0 = NULL, W0 = NULL, beta0 = NULL)
)
}
\arguments{
\item{x}{nxp matrix of p attributes observed for n objects.}

\item{k}{Number of distances to compute for each object (default: n-1).}

\item{D}{nxn or nxk dissimilarity matrix (optional). If absent, the Euclidean distance 
is computed.}

\item{J}{nxk matrix of indices. D[i,j] is the distance between objects i and
J[i,j]. (Used only if D is supplied and ncol(D)<n.)}

\item{c}{Number of clusters}

\item{type}{Type of focal sets ("simple": empty set, singletons and Omega;
"full": all \eqn{2^c} subsets of \eqn{\Omega}; "pairs": \eqn{\emptyset}, singletons,
\eqn{\Omega}, and all or selected pairs).}

\item{n_H}{Number of hidden units (if one hidden layer), or a two-dimensional vector
of numbers of hidden units (if two hidden layers).}

\item{ntrials}{Number of runs of the optimization algorithm (set to 1 if param0 is 
supplied).}

\item{d0}{Parameter used for matrix normalization. The normalized distance corresponding
to d0 is 0.95.}

\item{fhat}{Vector of outputs from a one-class SVM for novelty detection (optional)}

\item{lambda}{Regularization coefficient (default: 0)}

\item{y}{Optional vector of class labels for a subset of the training set 
(for semi-supervised learning).}

\item{Is}{Vector of indices corresponding to y (for semi-supervised learning).}

\item{nu}{Coefficient of the supervised error term (default: 0).}

\item{ML}{Optional nbML*2 matrix of must-link constraints (for constrained clustering). 
Each row of ML contains the indices of objects that belong to the same class.}

\item{CL}{Optional nbCL*2 matrix of cannot-link constraints (for constrained clustering). 
Each row of CL contains the indices of objects that belong to different classes.}

\item{xi}{Coefficient of the constrained clustering loss (default: 0).}

\item{tr}{If TRUE, a trace of the stress function is returned.}

\item{options}{Parameters of the optimization algorithm (see \code{\link{harris}}).}

\item{param0}{Optional list of initial network parameters (see details).}
}
\value{
The output credal partition (an object of class \code{"credpart"}). In 
addition to the usual attributes, the output credal partition has the following 
attributes:
 \describe{
  \item{Kmat}{The matrix of degrees of conflict. Same size as D.}
  \item{D}{The normalized dissimilarity matrix.}
  \item{trace}{Trace of the algorithm (Stress function vs iterations).}
  \item{J}{The matrix of indices.}
  \item{param}{The network parameters as a list with components U, V, W and beta.}
 }
}
\description{
\code{nnevclus} computes a credal partition from a dissimilarity matrix using the NN-EVCLUS
algorithm.
}
\details{
This is a neural network version of \code{kevclus}. The neural net has one or two layers
of ReLU units and a softmax output layer (see Denoeux, 2020). The weight matrices are 
denoted by U, V and W for a two-hidden-layer network, or V and W for a one-hidden-layer 
network. The inputs are a feature vector x, an optional distance matrix D, and an 
optional vector of one-class SVM outputs fhat, which is used for novelty detection. 
Part of the output belief mass is transfered to the empty set based on beta[1]+beta[2]*fhat,
where beta is an additional parameter vector. The network can be trained in fully
unsupervised mode, in semi-supervised mode (with class labels for a subset of the
learning instances), or with pairwise constraints. The output is a credal partition 
(a "credpart" object), with a specific field containing the network parameters (U, V, W,
beta).
}
\examples{
\dontrun{
## Example with one hidden layer and no novelty detection
data(fourclass)
x<-scale(fourclass[,1:2])
y<-fourclass[,3]
clus<-nnevclus(x,c=4,n_H=c(5,5),type='pairs') # One hidden layer
plot(clus,x,mfrow=c(2,2))

## Example with two hidden layers and novelty detection
library(kernlab)
data(fourclass)
x<-scale(fourclass[,1:2])
y<-fourclass[,3]
x<-data.frame(x)
svmfit<-ksvm(~.,data=x,type="one-svc",kernel="rbfdot",nu=0.2,kpar=list(sigma=0.2))
fhat<-predict(svmfit,newdata=x,type="decision")
clus<-nnevclus(x,k=200,c=4,n_H=c(5,5),type='pairs',fhat=fhat)
plot(clus,x,mfrow=c(2,2))

## Example with semi-supervised learning
data<-bananas(400)
x<-scale(data$x)
y<-data$y
Is<-sample(400, 50)  # Indices of labeled instances
plot(x,col=y,pch=y)
points(x[Is,],pch=16)
svmfit<-ksvm(~.,data=x,type="one-svc",kernel="rbfdot",nu=0.2,kpar=list(sigma=0.2))
fhat<-predict(svmfit,newdata=x,type="decision")
clus<-nnevclus(x,k=100,c=2,n_H=10,type='full',fhat=fhat,Is=Is,y=y[Is],nu=0.5)
plot(clus,x)

## Example with pairwise constraints
data<-bananas(400)
x<-scale(data$x)
y<-data$y
const<-create_MLCL(y,500)
clus<-nnevclus(x,k=100,c=2,n_H=10,type='full',fhat=fhat,ML=const$ML,CL=const$CL,
rho=0.5)
plot(clus,x)

## Example with pairwise constraints and PCCA
data(iris)
x<-scale(as.matrix(iris[,1:4]))
y<-as.integer(iris[,5])
const<-create_MLCL(y,100)
res.pcca<-pcca(x,3,const$ML,const$CL,beta=1)
plot(res.pcca$z,pch=y,col=y)
clus<-nnevclus(x=x,D=res.pcca$D,c=3,n_H=10,type='full',ML=const$ML,CL=const$CL,rho=0.5)
plot(clus,x[,3:4])
}

}
\references{
T. Denoeux. NN-EVCLUS: Neural Network-based Evidential Clustering. 
Information Sciences, Vol. 572, Pages 297-330, 2021.
}
\seealso{
\code{\link{nnevclus_mb}}, \code{\link{predict.credpart}}, 
\code{\link{kevclus}}, \code{\link{kcevclus}}, \code{\link{harris}}
}
\author{
Thierry Denoeux.
}
