#' Simulated independent and repeated measures two-way experiments over a set of sample sizes
#'
#' Wrapper for both independent and repeated measures two-way simulations. A vector of defined sample sizes is simulated under the
#' model provided.
#'
#' @param matrices_obj List - Output generated by `calculate_mean_matrix` that include cell mean and standard deviation matrices
#' @param nset Vector - If default values are used for both `balanced` and `group_size`, sample sizes to be used in simulations. If `balanced="FALSE"` and a matrix is provided to `group_size`, number to add to all elements of `group_size`.
#' @param balanced Logical - Whether the study will be performed with the same number of subjects in all groups. Default is `TRUE`. See 'details'.
#' @param group_size Matrix - Sample size for each condition (combination of factor levels). Only to be used when `balanced=FALSE`.
#' @param repeated_measurements Logical - Does the design have repeated measurements. Default is false.
#' @param loss Character - Type of selection of subjects in groups that have less observations than `max(group_size)`. Possible values are "random" and "sequential". Ignored if `repeated_measurements=FALSE` or `balanced=TRUE`. See 'details'.
#' @param distribution Character - Type of distribution to simulate. Possible values are 'normal', 'truncated.normal' or 'skewed'.
#' @param skewness Numeric - Momentum of distribution skewness, univariate distribution simulation.
#' @param shape Numeric - Degree of skewness in the distribution. May be a single value, have a length equal to the number of levels of any one of the factors or a length equal to the product of the length of each factor. For multivariate distribution simulations.
#' @param inferior_limit Numeric - Value of the lower bound for the truncated distribution, defaults to '-Inf'. Ignored if `distribution` is either "normal" or "skewed".
#' @param superior_limit Numeric - Value of the upper bound for the truncated distribution, defaults to 'Inf'. Ignored if `distribution` is either "normal" or "skewed".
#' @param nsims Integer - Number of iterations
#'
#'@details
#'
#' For unbalanced independent measures designs, this function generates a simulation with 'max(group_size)' for all combinations of factors and then eliminates observations
#' at random in those factor combinations that have less participants or study subjects. This is also the behavior for unbalanced repeated measures designs when `loss="random"`.
#'
#' For unbalanced repeated measures designs in which `loss="sequential"` the participants or subjects from the groups with less observations will be a subset of participants or
#' subjects of groups with more observations. The elimination strategy may not sound like the most efficient way to proceed, is quite fast anyhow.
#'
#' The 'n' column in the output will reflect how many observations each factor combination has. This should match the input matrix.
#'
#' @return List with of `data.frame`s of simulated outcome values under different sample sizes. Each data.frame includes factor level labels, iteration number and sample size.
#'
#' @examples
#' refmean <- 1
#' treatgroups <- 4
#' timepoints <- 5
#' treateff <- 1.5
#' timeeff <- 0.85
#' factors_levels_names <- list(treatment=letters[1:treatgroups], time=1:timepoints)
#' ## Independent design
#' effects_treat_time <- calculate_mean_matrix(refmean = refmean,
#'                                             fAeffect = treateff,fBeffect = timeeff,
#'                                             nlfA = treatgroups, nlfB = timepoints,
#'                                             label_list = factors_levels_names)
#' ## Inspect plot to check if matrices correspond to design
#' effects_treat_time$meansplot
#' n <- seq(from = 16, to = 24, by = 2)
#'
#' ## In this case, the default 'repeated_measurements', 'distribution' and options are used.
#' indep_simulation <- simulate_twoway_nrange(effects_treat_time, n)
#'
#' ## Simulate from a truncated distribution
#' indep_simulation_trunc <- simulate_twoway_nrange(matrices_obj = effects_treat_time, nset = n,
#'                           distribution="truncated.normal", inferior_limit= 0.8)
#'
#' ##randomly select iteration, select a condition
#' k <- sample(1:max(indep_simulation_trunc[[1]]$iteration), 1)
#' toviewdist <- indep_simulation_trunc[[1]]
#' toviewdist <- subset(toviewdist, iteration==k)
#' toviewdist <- subset(toviewdist, cond=="V6")
#' hist(toviewdist$y)
#'
#' @export
simulate_twoway_nrange <- function(matrices_obj, nset, balanced=TRUE, group_size=NULL, loss=NULL, repeated_measurements=FALSE, distribution="normal", skewness=1, shape=0, inferior_limit=-Inf, superior_limit=Inf, nsims=200)
{
  if(!repeated_measurements & (length(matrices_obj[[1]])==5 | length(matrices_obj)==5))
  {
    stop("It seems you have given a correlation value and a within factor as input to the 'calculate_mean_matrix' function.\nAre you sure you want to leave 'repeated_measurements' as 'FALSE'?")
  }
  if(repeated_measurements & ((length(matrices_obj[[1]])==2 & any(sapply(matrices_obj, ggplot2::is.ggplot))) | (length(matrices_obj)==2 & all(sapply(matrices_obj, is.matrix)))))
  {
    stop("It seems you have not given a correlation value and a within factor as input to the 'calculate_mean_matrix' function.\nAre you sure you want to set 'repeated_measurements' to 'TRUE'?")
  }
  if(balanced & length(group_size)==0)
  {
    testedns <- nset
  } else if (!balanced & length(group_size)>1)
  {
    testedns <- lapply(nset, "+", group_size)
  }
  if(!repeated_measurements & (length(matrices_obj[[1]])==2 | length(matrices_obj)==2))
  {
    message("Simulating independent observations experiment")
    sim_overens <- lapply(testedns, twoway_simulation_independent, matrices_obj, balanced=balanced, distribution=distribution, skewness=skewness, inferior_limit=inferior_limit, superior_limit=superior_limit, nsims=nsims)
  } else if (repeated_measurements & (length(matrices_obj[[1]])==5 | length(matrices_obj)==5))
  {
    message("Simulating repeated observations experiment")
    sim_overens <- lapply(testedns, twoway_simulation_correlated, matrices_obj, balanced=balanced, loss=loss, distribution=distribution, shape=shape, inferior_limit=inferior_limit, superior_limit=superior_limit, nsims=nsims)
  }
  sim_overens
}
