% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fcwt_batch.R
\name{fcwt_batch}
\alias{fcwt_batch}
\title{Fast continuous wavelet transform - Batch processing}
\usage{
fcwt_batch(
  signal,
  sample_freq,
  n_freqs,
  time_resolution,
  freq_begin = 2 * sample_freq/length(signal),
  freq_end = sample_freq/2,
  sigma = 1,
  max_batch_size = ceiling(4 * 10^9/(n_freqs * 4)),
  n_threads = 2L,
  progress_bar = FALSE
)
}
\arguments{
\item{signal}{Real-valued time series. The time steps are assumed to be evenly spaced.}

\item{sample_freq}{Sampling rate of input time series. This number primarily establishes
a connection to physical units which is used in other frequency definitions
as well as the units of the output data.}

\item{n_freqs}{Number of frequency bins generated by the CWT. The frequencies
are linearly distributed. Computation time increases when raising the
number of frequency bins.}

\item{time_resolution}{The time resolution in inverse units of \code{sample_freq} of the result.
Memory consumption is directly related to that.
Can not be higher than the time resolution of the input signal.}

\item{freq_begin, freq_end}{Optionally specifies the frequency range \verb{[freq_end, freq_begin]}. If not
specified the maximal meaningful frequency range, depending on the input signal,
is taken.
The range and \code{sample_freq} need to be specified in the same units.}

\item{sigma}{Sets a dimensionless parameter modifying the wavelet spread which in the
time-domain is roughly given by \eqn{\Sigma_t \sim \sqrt{2} \frac{\sigma}{f}}.
Changing this parameter adjusts the time/frequency uncertainty balance.
Defaults to 1.
Larger (lower) value of sigma corresponds to a better (worse) frequency
resolution and a worse (better) time resolution.}

\item{max_batch_size}{The maximal batch size that is used for splitting up the input sequence.
This limits the maximal memory that is used. Defaults to roughly 4GB.
The actual batch size is optimized for use with FFTW.}

\item{n_threads}{Number of threads used by the computation, if supported by your platform.
Defaults to 2 threads (to accomodate CRAN requirements).}

\item{progress_bar}{Monitoring progress can sometimes be useful when performing time consuming
operations. Setting \code{progress_bar = TRUE} enables printing a progress
bar to the console. Defaults to \code{FALSE}.}
}
\value{
The spectogram, a numeric real-valued matrix with dimensions roughly
\code{dim ~ c(length(signal) * time_resolution * sample_freq, n_freqs)}.
The exact length of the output depends on boundary effect details.
This matrix is wrapped into a S3-class \code{fcwtr_scalogram} so that plotting and
coercion functions can be used conveniently.
}
\description{
Performs a fast continuous wavelet transform on long sequences by sequentially
processing junks of the input signal and keeping only low-resolution output
data to preserve memory.
This is only useful for very long signals whose output does not fit into
memory as a whole.
It should not be used on short signals since boundary artefacts are discarded
(and those potentially dominate for short sequences).
}
\details{
In case of input sequences that exceed the a certain size, the output
sequence will not fit into the local memory and the fcwt cannot be
performed in one run.
For instance, in case of processing a song of 10 minutes length (assuming
a sampling rate of 44100 Hz), the size of the output vector is
\verb{10 * 60 seconds * 44100 Hz * nfreqs * 4 bytes},
which for e.g. \code{nfreqs = 200}, equals ~ 21 GB, hence
nowadays already at the limit of the hardware of a modern personal computer.

In cases where the required output time-resolution is smaller than the time
resolution of the input signal, one can perform the \code{\link[=fcwt]{fcwt()}} and reduce the
output size by averaging.
(The input signal time resolution can in general not be reduced since
high-frequency information would get lost.)

This function splits up the input sequence into batches, processes each
batch separately, reduces the time resolution, and adds the outputs together.

Attention: Boundary artefacts are removed, so some high frequency information
at the beginning and the end of the sequence is lost. (The amount depends on
the minimal frequency captured \code{min_freq}.)
}
\examples{
res <-
  fcwt_batch(
    ts_sin_sin,
    sample_freq = 44100,
    freq_begin = 100,
    freq_end = 11000,
    n_freqs = 10,
    sigma = 10,
    max_batch_size = 50000,
    time_resolution = 0.001
  )
}
\seealso{
\code{\link[=fcwt]{fcwt()}}
}
