% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/module.R
\name{fastGLCM_Rcpp}
\alias{fastGLCM_Rcpp}
\title{GLCM feature texture extraction}
\usage{
fastGLCM_Rcpp(
  data,
  methods,
  levels = 8,
  kernel_size = 5,
  distance = 1,
  angle = 0,
  dir_save = NULL,
  threads = 1,
  verbose = FALSE
)
}
\arguments{
\item{data}{a numeric matrix}

\item{methods}{a vector of character strings. One or all of the following: 'mean', 'std', 'contrast', 'dissimilarity', 'homogeneity', 'ASM', 'energy', 'max', 'entropy'}

\item{levels}{an integer specifying the window size. This parameter will create a mask of size \emph{levels x levels} internally}

\item{kernel_size}{an integer specifying the kernel size. A kernel of 1's will be created and the \emph{cv2.filter2D} filter will be utilized for the convolution}

\item{distance}{a numeric value specifying the pixel pair distance offsets (a 'pixel' value such as 1.0, 2.0 etc.)}

\item{angle}{a numeric value specifying the pixel pair angles (a 'degree' value such as 0.0, 30.0, 45.0, 90.0 etc.)}

\item{dir_save}{either NULL or a character string specifying a valid path to a directory where the output GLCM matrices (for the specified 'methods') will be saved. By setting this parameter to a valid directory the memory usage will be decreased.}

\item{threads}{an integer value specifying the number of cores to run in parallel}

\item{verbose}{a boolean. If TRUE then information will be printed out in the console}
}
\value{
a list consisting of one or more GLCM features
}
\description{
GLCM feature texture extraction
}
\details{
\strong{The following are two factors which (highly probable) will increase memory usage during computations:}
\itemize{
   \item \strong{1st.} the image size (the user might have to resize the image first)
   \item \strong{2nd.} the 'levels' parameter. The bigger this parameter the more matrices will be initialized and more memory will be used. For instance if the 'levels' parameter equals
   to 8 then 8 * 8 = 64 matrices of equal size to the input image will be initialized. That means if the image has dimensions (2745 x 2745) and the image-object size is approx. 60 MB then
   by initializing 64 matrices the memory will increase to 3.86 GB.
}

\strong{This function is an Rcpp implementation} of the python fastGLCM module. When using each function separately by utilizing all threads it's slightly faster compared to the python
vectorized functions, however it's a lot faster when computing all features at once.

The \strong{dir_save} parameter allows the user to save the GLCM's as .csv files to a directory. That way the output GLCM's matrices won't be returned in the R session (reduced memory usage). However, by
saving the GLCM's to .csv files the computation time increases.
}
\examples{

require(fastGLCM)
require(OpenImageR)
require(utils)

temp_dir = tempdir(check = FALSE)
# temp_dir

zip_file = system.file('images', 'JAXA_Joso-City2_PAN.tif.zip', package = "fastGLCM")
utils::unzip(zip_file, exdir = temp_dir)
path_extracted = file.path(temp_dir, 'JAXA_Joso-City2_PAN.tif')

im = readImage(path = path_extracted)
dim(im)

#...............................................
# resize the image and adjust pixel values range
#...............................................

im = resizeImage(im, 500, 500, 'nearest')
im = OpenImageR::norm_matrix_range(im, 0, 255)

#---------------------------------
# computation of all GLCM features
#---------------------------------

methods = c('mean',
            'std',
            'contrast',
            'dissimilarity',
            'homogeneity',
            'ASM',
            'energy',
            'max',
            'entropy')

res_glcm = fastGLCM_Rcpp(data = im,
                         methods = methods,
                         levels = 8,
                         kernel_size = 5,
                         distance = 1.0,
                         angle = 0.0,
                         threads = 1)
# str(res_glcm)

# plot_multi_images(list_images = res_glcm,
#                   par_ROWS = 2,
#                   par_COLS = 5,
#                   titles = methods)

if (file.exists(path_extracted)) file.remove(path_extracted)
}
\references{
https://github.com/tzm030329/GLCM
}
