#' @title Predict Method for fastNaiveBayes fits
#' @description Uses a fastNaiveBayes model and a new data set to create the classifications.
#'     This can either be the raw probabilities generated by the fastNaiveBayes model or the classes themselves.
#'
#' @param object A fitted object of class "fnb.bernoulli", "fnb.gaussian", "fnb.multinomial" or "fastNaiveBayes".
#' @param newdata A numeric matrix. A Sparse dgcMatrix is also accepted.
#'     Note: if fnb.naiveBayes or fnb.gaussian or fnb.train was used to create the model, then if newdata contains features that were not encountered in the training data, these are omitted from the prediction.
#'     Furthermore, newdata can contain fewer features than encountered in the training data. In this case, newdata will be padded with
#'     extra columns all filled with 0's.
#' @param type If "raw", the conditional a-posterior probabilities for each class are returned, and the class with maximal probability else.
#' @param sparse Use a sparse matrix? If true a sparse matrix will be constructed from x.
#'     It's possible to directly feed a sparse dgcMatrix as x, which will set this parameter to TRUE
#' @param threshold A threshold for the minimum probability. For Bernoulli and Multinomial event models Laplace smoothing should solve this,
#' but in the case of Gaussian event models, this ensures numerical probabilities.
#' @param check Whether to perform formal checks on the input. Set to false, if this is not necessary and speed is of the essence
#' @param ... Not used.
#' @return If type = 'class', a factor with classified class levels. If type = 'raw', a matrix with the predicted probabilities of
#'     each class, where each column in the matrix corresponds to a class level.
#'
#' @export
#' @import Matrix
#'
#' @details In the extremely unlikely case that two classes have the exact same estimated probability, the first encountered class
#'     is used as the classification.
#'
#'     Using a sparse matrix directly can be especially useful if it's necessary to use predict multiple times on the same matrix or
#'     on different subselections of the same initial matrix, see examples for further details.
#' @examples
#' rm(list = ls())
#' library(fastNaiveBayes)
#' cars <- mtcars
#' y <- as.factor(ifelse(cars$mpg > 25, "High", "Low"))
#' x <- cars[, 2:ncol(cars)]
#'
#' dist <- fnb.detect_distribution(x, nrows = nrow(x))
#' print(dist)
#'
#' # Mixed event models
#' mod <- fnb.train(x, y, laplace = 1) # alternatively: fastNaiveBayes(x, y, laplace = 1)
#' pred <- predict(mod, newdata = x)
#' mean(pred != y)
#'
#' # Bernoulli only
#' vars <- c(dist$bernoulli, dist$multinomial)
#' newx <- x[, vars]
#' for (i in 1:ncol(newx)) {
#'   newx[[i]] <- as.factor(newx[[i]])
#' }
#' new_mat <- model.matrix(y ~ . - 1, cbind(y, newx))
#'
#' mod <- fnb.bernoulli(new_mat, y, laplace = 1)
#' pred <- predict(mod, newdata = new_mat)
#' mean(pred != y)
#'
#' # Multinomial only
#' vars <- c(dist$bernoulli, dist$multinomial)
#' newx <- x[, vars]
#'
#' mod <- fnb.multinomial(newx, y, laplace = 1)
#' pred <- predict(mod, newdata = newx)
#' mean(pred != y)
#'
#' # Gaussian only
#' vars <- c("hp", dist$gaussian)
#' newx <- x[, vars]
#'
#' mod <- fnb.gaussian(newx, y)
#' pred <- predict(mod, newdata = newx)
#' mean(pred != y)
#'
#' @seealso \code{\link{fastNaiveBayes}} for the fastNaiveBayes model
#' @rdname predict.fastNaiveBayes
predict.fastNaiveBayes <- function(object, newdata, type = c("class", "raw"), sparse = FALSE,
                                   threshold = .Machine$double.eps, check = TRUE, ...) {

  type <- match.arg(type)
  if(check){
    args <- fnb.check.args.predict(object, newdata, type, sparse, threshold, ...)
    object <- args$object
    newdata <- args$newdata
    type <- args$type
    sparse <- args$sparse
    threshold <- args$threshold
  }

  probs <- NULL
  for (i in 1:length(object$models)) {
    model <- object$models[[i]]
    if (is.null(probs)) {
      probs <- stats::predict(model, newdata, type = "rawprob", sparse, threshold, silent = TRUE)
    } else {
      probs <- probs + stats::predict(model, newdata, type = "rawprob", sparse, threshold, silent = TRUE)
    }
  }
  probs <- exp(probs)

  priors <- object$priors
  for(i in 1:length(priors)){
    probs[,i] <- probs[,i]*priors[i]
  }

  denom <- rowSums(probs)
  denom[denom==0] <- 1
  probs <- probs / denom

  if (type == "class") {
    class <- as.factor(object$levels[max.col(probs, ties.method = "first")])
    levels(class) <- object$levels
    return(class)
  }
  return(probs)
}
