\name{fdt}
\alias{fdt}
\alias{fdt.default}
\alias{fdt.data.frame}
\alias{fdt.matrix}

\title{
  Frequency Distribution Table
}

\description{
  A S3 set of methods to easily perform frequency distribution table (fdt) from
  \code{vector}, \code{data.frame} and \code{matrix} objects.
}

\usage{
  fdt(x, \dots)

  \method{fdt}{default}(x, k, start, end, h, breaks=c("Sturges", "Scott", "FD"),
    right=FALSE, \dots)
  \method{fdt}{data.frame}(x, k, by, breaks=c("Sturges", "Scott", "FD"),
    right=FALSE, \dots)
  \method{fdt}{matrix}(x, k, breaks=c("Sturges", "Scott", "FD"),
    right=FALSE, \dots)
}

\arguments{
  \item{x}{A numeric \code{vector}, \code{data.frame} or \code{matrix} object.
    If \code{x} is \code{data.frame} or \code{matrix} it must contain at least 
    one numeric column.}
  \item{k}{Number of class intervals.}
  \item{start}{Left endpoint of the first class interval.}
  \item{end}{Right endpoint of the last class interval.}
  \item{h}{Class interval width.}
  \item{by}{Categorical variable used for grouping each numeric variable, useful
    only on \code{data.frames}.}
  \item{breaks}{Method used to determine the number of interval classes, 
    c("Sturges", "Scott", "FD").}
  \item{right}{Right endpoints open (default = FALSE).}
  \item{\dots}{Potencial further arguments (required by generic).}
}

\details{
  The simplest way to run \code{fdt} is done by supplying only the \code{x}
  object, for example: \code{d <- fdt(x)}. In this case all necessary
  default values (\code{breaks} and \code{right}) ("Sturges" and \code{FALSE}
  respectivelly) will be used.

  It can also be provided: a) \code{x} and \code{k}; b) \code{x}, \code{start}
  and \code{end}; or c) \code{x}, \code{start}, \code{end} and \code{h}. These
  options make the \code{fdt} very easy and flexible to use.

  The \code{fdt} object stores information to be used by methods \code{summary},
  \code{print} and \code{plot}. The result of plot is a histogram.
  The methods \code{summary}, \code{print} and \code{plot} provide a reasonable
  set of parameters to format and plot the \code{fdt} object in a pretty
  (and publishable) way.
}

\value{
  The method \code{fdt.default} returns a list of class \code{fdt.default}
  with the slots:
  \item{table}{A \code{data.frame} storing the fdt.}
  \item{breaks}{A \code{vector} of length 4 storing \code{start}, \code{end},
    \code{h} and \code{right} of the fdt generated by this method.}
  \item{data}{A vector of the data \code{x} provided.}

  The methods \code{fdt.data.frame} and \code{fdt.matrix} return a list of class 
  \code{fdt.multiple}. This \code{list} has one slot for each numeric
  variable of the \code{x} provided. Each slot, corresponding to each numeric
  variable, stores the same slots of the \code{fdt.default} described above.
}

\author{
  Jose Claudio Faria (\email{joseclaudio.faria@gmail.com})\cr
  Enio Jelihovschi (\email{eniojelihovs@gmail.com})\cr
}

\seealso{
 \code{\link[graphics]{hist}} provided by \code{graphics};
 \code{\link[base]{table}}, \code{\link[base]{cut}} both provided by \code{base}
 and \code{\link[Hmisc]{hist.data.frame}} provided by \code{Hmisc} package.
}

\examples{
library(fdth)

#======================
# Vectors: univariated
#======================
set.seed(1); x <- rnorm(n=1e3, mean=5, sd=1)

# x
d <- fdt(x); d

# x, alternative breaks
d <- fdt(x, breaks='Scott'); d

# x, k
d <- fdt(x, k=20); d

# x, star, end
range(x)
d <- fdt(x, start=1.5, end=9); d

# x, start, end, h
d <- fdt(x, start=1, end=9, h=1); d

# Effect of right
x <- rep(1:3, 3); sort(x)
d <- fdt(x, start=1, end=4, h=1); d

d <- fdt(x, start=0, end=3, h=1, right=TRUE); d

#=============================================
# Data.frames: multivariated with categorical
#=============================================
mdf <- data.frame(X1 = rep(LETTERS[1:4], 25),
                  X2 = as.factor(rep(1:10, 10)),
                  Y1 = c(NA, NA, rnorm(96, 10, 1), NA, NA),
                  Y2 = rnorm(100, 60, 4),
                  Y3 = rnorm(100, 50, 4),
                  Y4 = rnorm(100, 40, 4))

d <- fdt(mdf); d

levels(mdf$X1)
d <- fdt(mdf, k=5, by='X1'); d

d <- fdt(mdf, breaks='FD', by='X1')
str(d)
d

levels(mdf$X2)
d <- fdt(mdf, breaks='FD', by='X2'); d

d <- fdt(mdf, k=5, by='X2'); d

d <- fdt(iris, k=5); d

d <- fdt(iris, k=10); d

levels(iris$Species)
d <- fdt(iris, k=5, by='Species'); d

require(MASS)
levels(Cars93$Origin)
d <- fdt(Cars93, k=5, by='Origin'); d

d <- fdt(Cars93, breaks='FD', by='Origin'); d

#=========================
# Matrices: multivariated
#=========================
d <-fdt(state.x77); d

d <-fdt(volcano); d
}

\keyword{fdt}
\keyword{frequency}
\keyword{distribution}
\keyword{table}
