% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods.R
\name{summary.fixest}
\alias{summary.fixest}
\alias{summary.fixest_list}
\title{Summary of a \code{fixest} object. Computes different types of standard errors.}
\usage{
\method{summary}{fixest}(
  object,
  vcov = NULL,
  cluster = NULL,
  ssc = NULL,
  .vcov = NULL,
  stage = NULL,
  lean = FALSE,
  agg = NULL,
  forceCovariance = FALSE,
  se = NULL,
  keepBounded = FALSE,
  n = 1000,
  vcov_fix = TRUE,
  nthreads = getFixest_nthreads(),
  ...
)

\method{summary}{fixest_list}(
  object,
  se,
  cluster,
  ssc = getFixest_ssc(),
  .vcov,
  stage = 2,
  lean = FALSE,
  n,
  ...
)
}
\arguments{
\item{object}{A \code{fixest} object. Obtained using the functions \code{\link{femlm}}, \code{\link{feols}} or \code{\link{feglm}}.}

\item{vcov}{Versatile argument to specify the VCOV. In general, it is either a character
scalar equal to a VCOV type, either a formula of the form: \code{vcov_type ~ variables}. The
VCOV types implemented are: "iid", "hetero" (or "HC1"), "cluster", "twoway",
"NW" (or "newey_west"), "DK" (or "driscoll_kraay"), and "conley". It also accepts
object from \code{\link{vcov_cluster}}, \code{\link[=vcov_hac]{vcov_NW}}, \code{\link[=vcov_hac]{NW}},
\code{\link[=vcov_hac]{vcov_DK}}, \code{\link[=vcov_hac]{DK}}, \code{\link{vcov_conley}} and
\code{\link[=vcov_conley]{conley}}. It also accepts covariance matrices computed externally.
Finally it accepts functions to compute the covariances. See the \code{vcov} documentation
in the \href{https://lrberge.github.io/fixest/articles/fixest_walkthrough.html#the-vcov-argument-1}{vignette}.}

\item{cluster}{Tells how to cluster the standard-errors (if clustering is requested).
Can be either a list of vectors, a character vector of variable names, a formula or
an integer vector. Assume we want to perform 2-way clustering over \code{var1} and \code{var2}
contained in the data.frame \code{base} used for the estimation. All the following
\code{cluster} arguments are valid and do the same thing:
\code{cluster = base[, c("var1", "var2")]}, \code{cluster = c("var1", "var2")}, \code{cluster = ~var1+var2}.
If the two variables were used as fixed-effects in the estimation, you can leave it
blank with \code{vcov = "twoway"} (assuming \code{var1} [resp. \code{var2}] was
the 1st [resp. 2nd] fixed-effect). You can interact two variables using \code{^} with
the following syntax: \code{cluster = ~var1^var2} or \code{cluster = "var1^var2"}.}

\item{ssc}{An object of class \code{ssc.type} obtained with the function \code{\link{ssc}}. Represents
how the degree of freedom correction should be done.You must use the function \code{\link{ssc}}
for this argument. The arguments and defaults of the function \code{\link{ssc}} are:
\code{adj = TRUE}, \code{fixef.K="nested"}, \code{cluster.adj = TRUE}, \code{cluster.df = "min"},
\code{t.df = "min"}, \verb{fixef.force_exact=FALSE)}. See the help of the function \code{\link{ssc}} for details.}

\item{.vcov}{A user provided covariance matrix or a function computing this matrix. If a
matrix, it must be a square matrix of the same number of rows as the number
of variables estimated. If a function, it must return the previously mentioned matrix.}

\item{stage}{Can be equal to \code{2} (default), \code{1}, \code{1:2} or \code{2:1}. Only used if the object
is an IV estimation: defines the stage to which \code{summary} should be applied. If \code{stage = 1}
and there are multiple endogenous regressors or if \code{stage} is of length 2, then an
object of class \code{fixest_multi} is returned.}

\item{lean}{Logical, default is \code{FALSE}. Used to reduce the (memory) size of the summary object.
If \code{TRUE}, then all objects of length N (the number of observations) are removed
from the result. Note that some \code{fixest} methods may consequently not work when applied
to the summary.}

\item{agg}{A character scalar describing the variable names to be aggregated,
it is pattern-based. For \code{\link{sunab}} estimations, the following keywords work: "att",
"period", "cohort" and \code{FALSE} (to have full disaggregation). All variables that
match the pattern will be aggregated. It must be of the form \code{"(root)"}, the parentheses
must be there and the resulting variable name will be \code{"root"}. You can add another
root with parentheses: \code{"(root1)regex(root2)"}, in which case the resulting
name is \code{"root1::root2"}. To name the resulting variable differently you can pass
a named vector: \code{c("name" = "pattern")} or \code{c("name" = "pattern(root2)")}. It's a
bit intricate sorry, please see the examples.}

\item{forceCovariance}{(Advanced users.) Logical, default is \code{FALSE}. In the peculiar case
where the obtained Hessian is not invertible (usually because of collinearity of
some variables), use this option to force the covariance matrix, by using a generalized
inverse of the Hessian. This can be useful to spot where possible problems come from.}

\item{se}{Character scalar. Which kind of standard error should be computed:
\dQuote{standard}, \dQuote{hetero}, \dQuote{cluster}, \dQuote{twoway}, \dQuote{threeway}
or \dQuote{fourway}? By default if there are clusters in the estimation:
\code{se = "cluster"}, otherwise \code{se = "iid"}. Note that this argument is deprecated,
you should use \code{vcov} instead.}

\item{keepBounded}{(Advanced users -- \code{feNmlm} with non-linear part and bounded
coefficients only.) Logical, default is \code{FALSE}. If \code{TRUE}, then the bounded coefficients
(if any) are treated as unrestricted coefficients and their S.E. is computed (otherwise
it is not).}

\item{n}{Integer, default is 1000. Number of coefficients to display when the print method
is used.}

\item{vcov_fix}{Logical scalar, default is \code{TRUE}. If the VCOV ends up not being
positive definite, whether to "fix" it using an eigenvalue decomposition
(a la Cameron, Gelbach & Miller 2011).}

\item{nthreads}{The number of threads. Can be: a) an integer lower than, or equal to,
the maximum number of threads; b) 0: meaning all available threads will be used;
c) a number strictly between 0 and 1 which represents the fraction of all threads to use.
The default is to use 50\% of all threads. You can set permanently the number
of threads used within this package using the function \code{\link{setFixest_nthreads}}.}

\item{...}{Only used if the argument \code{.vcov} is provided and is a function: extra arguments
to be passed to that function.}
}
\value{
It returns a \code{fixest} object with:
\item{cov.scaled}{The new variance-covariance matrix (computed according to the argument \code{se}).}
\item{se}{The new standard-errors (computed according to the argument \code{se}).}
\item{coeftable}{The table of coefficients with the new standard errors.}
}
\description{
This function is similar to \code{print.fixest}. It provides the table of coefficients along with
other information on the fit of the estimation. It can compute different types of standard
errors. The new variance covariance matrix is an object returned.
}
\section{Compatibility with \pkg{sandwich} package}{

The VCOVs from \code{sandwich} can be used with \code{feols}, \code{feglm} and \code{fepois} estimations.
If you want to have a \code{sandwich} VCOV when using \code{summary.fixest}, you can use
the argument \code{vcov} to specify the VCOV function to use (see examples).
Note that if you do so and you use a formula in the \code{cluster} argument, an innocuous
warning can pop up if you used several non-numeric fixed-effects in the estimation
(this is due to the function \code{\link{expand.model.frame}} used in \code{sandwich}).
}

\examples{

# Load trade data
data(trade)

# We estimate the effect of distance on trade (with 3 fixed-effects)
est_pois = fepois(Euros ~ log(dist_km)|Origin+Destination+Product, trade)

# Comparing different types of standard errors
sum_standard = summary(est_pois, vcov = "iid")
sum_hetero   = summary(est_pois, vcov = "hetero")
sum_oneway   = summary(est_pois, vcov = "cluster")
sum_twoway   = summary(est_pois, vcov = "twoway")

etable(sum_standard, sum_hetero, sum_oneway, sum_twoway)

# Alternative ways to cluster the SE:
summary(est_pois, vcov = cluster ~ Product + Origin)
summary(est_pois, vcov = ~Product + Origin)
summary(est_pois, cluster = ~Product + Origin)

# You can interact the clustering variables "live" using the var1 ^ var2 syntax.#'
summary(est_pois, vcov = ~Destination^Product)

#
# Newey-West and Driscoll-Kraay SEs
#

data(base_did)
# Simple estimation on a panel
est = feols(y ~ x1, base_did)

# --
# Newey-West
# Use the syntax NW ~ unit + time
summary(est, NW ~ id + period)

# Now take a lag of 3:
summary(est, NW(3) ~ id + period)

# --
# Driscoll-Kraay
# Use the syntax DK ~ time
summary(est, DK ~ period)

# Now take a lag of 3:
summary(est, DK(3) ~ period)

#--
# Implicit deductions
# When the estimation is done with a panel.id, you don't need to
# specify these values.

est_panel = feols(y ~ x1, base_did, panel.id = ~id + period)

# Both methods, NM and DK, now work automatically
summary(est_panel, "NW")
summary(est_panel, "DK")

#
# VCOVs robust to spatial correlation
#

data(quakes)
est_geo = feols(depth ~ mag, quakes)

# --
# Conley
# Use the syntax: conley(cutoff) ~ lat + lon
# with lat/lon the latitude/longitude variable names in the data set
summary(est_geo, conley(100) ~ lat + long)

# Change the cutoff, and how the distance is computed
summary(est_geo, conley(200, distance = "spherical") ~ lat + long)

# --
# Implicit deduction
# By default the latitude and longitude are directly fetched in the data based
# on pattern matching. So you don't have to specify them.
# Further an automatic cutoff is computed by default.

# The following works
summary(est_geo, "conley")



#
# Compatibility with sandwich
#

# You can use the VCOVs from sandwich by using the argument .vcov:
library(sandwich)
summary(est_pois, .vcov = vcovCL, cluster = trade[, c("Destination", "Product")])


}
\seealso{
See also the main estimation functions \code{\link{femlm}}, \code{\link{feols}} or \code{\link{feglm}}.
Use \code{\link{fixef.fixest}} to extract the fixed-effects coefficients, and the function \code{\link{etable}}
to visualize the results of multiple estimations.
}
\author{
Laurent Berge
}
