% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/item_estimation.R
\name{fmp}
\alias{fmp}
\alias{fmp_1}
\alias{fmp}
\title{Estimate FMP Item Parameters}
\usage{
fmp_1(dat, k, tsur, start_vals = NULL, method = "BFGS", ...)

fmp(dat, k, start_vals = NULL, em = TRUE, eps = 1e-04, n_quad = 49,
  method = "BFGS", max_em = 500, ...)
}
\arguments{
\item{dat}{Vector of 0/1 item responses for N (# subjects) examinees.}

\item{k}{Vector of item complexities for each item, see details.}

\item{tsur}{Vector of N (# subjects) surrogate theta values.}

\item{start_vals}{Start values, For fmp_1, a vector of length 2k+2 in the
following order:

If k = 0: (xi, omega)

If k = 1: (xi, omega, alpha1, tau1)

If k = 2: (xi, omega, alpha1, tau1, alpha2, tau2)

and so forth. For fmp, add start values for item 1, followed by those for
item 2, and so forth. For further help, first fit the model without start
values, then inspect the outputted parmat data frame.}

\item{method}{Optimization method passed to optim.}

\item{em}{Logical, use random-effects estimation using the EM algorith? If
FALSE, fixed effects estimation is used with theta surrogates.}

\item{eps}{Covergence tolerance for the EM algorithm. The EM algorithm is
said to converge is the maximum absolute difference between parameter
estimates for successive iterations is less than eps. Ignored if em = FALSE.}

\item{n_quad}{Number of quadrature points for EM integration. Ignored if
em = FALSE}

\item{max_em}{Maximum number of EM iterations.}

\item{\ldots}{Additional arguments passed to optim.}
}
\value{
\item{bmat}{Matrix of estimated b-matrix parameters, each row corresponds
to an item, and contains b0, b1, ...b(max(k)).}
\item{parmat}{Data frame of parameter estimation information, including the
Greek-letter parameterization, starting value, and parameter estimate.}
\item{k}{Vector of item complexities chosen for each item.}
\item{log_lik}{Model log likelihood.}
\item{mod}{Optimization information, including output from optim.}
\item{AIC}{Model AIC.}
\item{BIC}{Model BIC.}
}
\description{
Estimate FMP item parameters for a single item using user-specified
theta values (fixed-effects) using fmp_1, or estimate FMP item parameters
for multiple items using fixed-effects or random-effects with fmp.
}
\details{
The FMP item response function for a single item is specified using
the composite function,
\deqn{P(\theta)=[1+\exp(-m(\theta))]^{-1},}{P(\theta)=[1+exp(-m(\theta))]^{-1},}
where \eqn{m(\theta)} is an unbounded and monotonically
increasing polynomial function of the latent trait \eqn{\theta}.

The item complexity parameter \eqn{k} controls the degree of the polynomial:
\deqn{m(\theta)=b_0+b_1\theta+b_2\theta^{2}+...+b_{2k+1}
\theta^{2k+1},}{m(\theta)=b0+b1\theta+b2\theta^{2}+...+b(2k+1)\theta^{2k+1},}
where \eqn{2k+1} equals the order of the polynomial,
\eqn{k} is a nonnegative integer, and
\deqn{b=(b0,b1,...,b(2k+1))'}
are item parameters that define the location and shape of the IRF. The
vector \eqn{b} is called the b-vector parameterization of the FMP Model.
When \eqn{k=0}, the FMP IRF equals
\deqn{P(\theta)= [1+\exp(-(b_0+b_1\theta))] ^{-1},}{P(\theta)=
[1+exp(-(b0+b1\theta))] ^{-1},}
and is equivalent to the slope-threshold parameterization of the
two-parameter item response model.

For \eqn{m(\theta)} to be a monotonic function, the FMP IRF can also be
expressed as a function of the vector

\deqn{\gamma = (\xi, \omega, \alpha_1, \tau_1, \alpha_2, \tau_2,
\cdots \alpha_k,\tau_k)'.}{\gamma = (\xi, \omega, \alpha1, \tau1, \alpha2,
\tau2, ... \alpha_k, \tau_k)'.}

The \eqn{\gamma} vector is called the Greek-letter parameterization of the
FMP model. See Feuerstahler (2016) or Liang & Browne (2015) for details
about the relationship between the b-vector and Greek-letter
parameterizations.
}
\examples{

set.seed(2342)
bmat <- sim_bmat(n_items = 5, k = 2)$bmat

theta <- rnorm(50)
dat <- sim_data(bmat = bmat, theta = theta)

## fixed-effects estimation for item 1

tsur <- get_surrogates(dat)

# k = 0
fmp0_it_1 <- fmp_1(dat = dat[, 1], k = 0, tsur = tsur)

# k = 1
fmp1_it_1 <- fmp_1(dat = dat[, 1], k = 1, tsur = tsur)

# k = 2
fmp2_it_1 <- fmp_1(dat = dat[, 1], k = 2, tsur = tsur)


## fixed-effects estimation for all items

fmp0_fixed <- fmp(dat = dat, k = 0, em = FALSE)

## random-effects estimation for all items

fmp0_random <- fmp(dat = dat, k = 0, em = TRUE)

}
\references{
Elphinstone, C. D. (1983). A target distribution model for nonparametric
density estimation. \emph{Communication in Statistics--Theory
and Methods}, \emph{12}, 161--198. \doi{10.1080/03610928308828450}

Elphinstone, C. D. (1985). \emph{A method of distribution and density
estimation} (Unpublished dissertation). University of South Africa,
Pretoria, South Africa. \doi{20.500.11892/132832}

Falk, C. F., & Cai, L. (2016a). Maximum marginal likelihood estimation of a
monotonic polynomial generalized partial credit model with applications to
multiple group analysis. \emph{Psychometrika}, \emph{81}, 434--460.
\doi{10.1007/s11336-014-9428-7}

Falk, C. F., & Cai, L. (2016b). Semiparametric item response functions in
the context of guessing. \emph{Journal of Educational Measurement},
\emph{53}, 229--247. \doi{10.1111/jedm.12111}

Feuerstahler, L. M. (2016). \emph{Exploring alternate latent trait metrics
with the filtered monotonic polynomial IRT model} (Unpublished dissertation).
University of Minnesota, Minneapolis, MN. \url{http://hdl.handle.net/11299/182267}

Liang, L. (2007). \emph{A semi-parametric approach to estimating item
response functions} (Unpublished dissertation). The Ohio
State University, Columbus, OH. Retrieved from https://etd.ohiolink.edu/

Liang, L., & Browne, M. W. (2015). A quasi-parametric method for
fitting flexible item response functions. \emph{Journal of Educational
and Behavioral Statistics}, \emph{40}, 5--34. \doi{10.3102/1076998614556816}
}
