% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/powerPlot.R
\name{powerPlot}
\alias{powerPlot}
\title{Exclusion/inclusion power plots}
\usage{
powerPlot(
  ep,
  ip = NULL,
  type = 1,
  majorpoints = TRUE,
  minorpoints = TRUE,
  ellipse = FALSE,
  col = NULL,
  labs = NULL,
  jitter = FALSE,
  alpha = 1,
  stroke = 1.5,
  shape = "circle",
  size = 1,
  hline = NULL,
  vline = NULL,
  xlim = NULL,
  ylim = NULL,
  xlab = NULL,
  ylab = NULL
)
}
\arguments{
\item{ep, ip}{Lists of equal length, with outputs from one or more runs of
\code{\link[=missingPersonEP]{missingPersonEP()}} and \code{\link[=missingPersonIP]{missingPersonIP()}} respectively. Alternatively,
\code{ep} can be a single output from \code{\link[=MPPsims]{MPPsims()}}, in which case \code{ip} should be
NULL. See Examples.}

\item{type}{Plot type; either 1, 2, 3 or 4.}

\item{majorpoints}{A logical indicating whether "major" points should be
drawn (see Details).}

\item{minorpoints}{A logical indicating whether "minor" points should be
drawn (see Details).}

\item{ellipse}{A logical. If TRUE, data ellipsis are drawn for each group
containing more than 1 element. NB: This fails with a warning if all points
in a group fall on a line.}

\item{col}{A colour vector, recycle to match the top level length of \code{ep}.}

\item{labs}{A character of the same length as \code{ep}. If NULL, the names of
\code{ep} are used, if present.}

\item{jitter}{A logical (default: FALSE). If TRUE, a small jitter is added to
the major points.}

\item{alpha}{Transparency for minor points (see Details).}

\item{stroke}{Border width for major points (see Details).}

\item{shape}{Either "circle", "square", "diamond", "triangleUp" or
"triangleDown", determining the shapes of both minor and major points.}

\item{size}{Point size.}

\item{hline, vline}{Single numbers indicating positions for
horizontal/vertical "threshold" lines. If NULL (default), no lines are
drawn.}

\item{xlim, ylim}{Axis limits; automatically chosen if NULL.}

\item{xlab, ylab}{Axis labels; automatically chosen if NULL.}
}
\value{
A \code{ggplot2} plot object.
}
\description{
This function offers four different visualisations of exclusion/inclusion
powers, particularly for missing person cases. Output from \code{\link[=MPPsims]{MPPsims()}} may be
fed directly as input to this function. The actual plotting is done with
\code{ggplot2}.
}
\details{
The plot types are as follows:

\code{type = 1}: x = Exclusion power; y = Inclusion power

\code{type = 2}: x = Exclusion odds ratio; y = Inclusion odds ratio

\code{type = 3}: x = Expected number of exclusions; y = average log(LR)

\code{type = 4}: x = Exclusion power; y = average LR

In the most general case \code{ep} (and similarly for \code{ip}) can be a list of lists
of \code{EPresult} objects. We refer to the inner lists as "groups". A group may
consist of a single output, or several (typically many simulations of the
same situation). Points within the same group are always drawn with the same
colour and shape.

When plotting several groups, two sets of points are drawn by default:
\itemize{
\item Major points: Group means.
\item Minor points: Individual points in groups with more than one element.
}

The parameters \code{majorpoints} and \code{minorpoints} control which of the above
points are included.
}
\examples{

### Example 1: Comparing the power of 3 reference families ###

# Helper function for simulating a single profile
sim = function(x, id) {
  y = setMarkers(x, locusAttributes = NorwegianFrequencies[1:3])
  profileSim(y, ids = id)[[1]]
}

# Define pedigrees and simulate data
PAR = nuclearPed(1, child = "MP") |> sim(id = 1)
SIB = nuclearPed(2) |> relabel(old = 4, new = "MP") |> sim(id = 3)
GRA = linearPed(2) |> relabel(old = 5, new = "MP") |> sim(id = 1)

# Collect in list and plot
peds = list(PAR = PAR, SIB = SIB, GRA = GRA)
plotPedList(peds, marker = 1, hatched = typedMembers, frames = FALSE,
            col = list(red = "MP"))

# Compute exclusion/inclusion powers:
ep = lapply(peds, function(y)
  missingPersonEP(y, missing = "MP", verbose = FALSE))

ip = lapply(peds, function(y)    # increase nsim!
  missingPersonIP(y, missing = "MP", nsim = 5, threshold = 10, verbose = FALSE))

# Plot
powerPlot(ep, ip, size = 2)
powerPlot(ep, ip, size = 2, jitter = TRUE)

# Different plot type, not dependent of `threshold`
powerPlot(ep, ip, size = 2, type = 3)

### Example 2: Exploring powers for different sets of available relatives

# Create trio pedigree
ref = nuclearPed(father = "fa", mother = "mo", child = "MP")

# Add empty marker with 5 alleles
ref = addMarker(ref, alleles = 1:5)

# Alternatives for genotyping
sel = list("fa", c("fa", "mo"))

# Simulate power for each selection
simData = MPPsims(ref, selections = sel, nProfiles = 3, lrSims = 5,
                  thresholdIP = 2, seed = 123, numCores = 1)

# Power plot 1: EP vs IP
powerPlot(simData, type = 1)
powerPlot(simData, type = 1, minorpoints = FALSE, hline = 0.8)

\donttest{

# Change shape, and modify legend order
powerPlot(simData[3:1], type = 1, shape = c("ci", "sq", "di"))

# Zoom in, and add threshold lines
powerPlot(simData, type = 1, xlim = c(0.2, 1), ylim = c(0.5, 1),
          hline = 0.8, vline = 0.8)

# Power plot 3: Expected number of exclusions vs E[log LR]
powerPlot(simData, type = 3)

# With horizontal/vertical lines
powerPlot(simData, type = 3, hline = log10(2), vline = 1)

# Plot 4: Illustrating the general inequality ELR > 1/(1-EP)
powerPlot(simData, type = 4)

}

}
\seealso{
\code{\link[=MPPsims]{MPPsims()}}, \code{\link[=missingPersonEP]{missingPersonEP()}}, \code{\link[=missingPersonEP]{missingPersonEP()}}
}
