% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/funDataMethods.R
\name{scalarProduct}
\alias{scalarProduct}
\title{Calculate the scalar product for functional data objects}
\usage{
scalarProduct(object1, object2, ...)
}
\arguments{
\item{object1, object2}{Two objects of class\code{\link{funData}}, 
\code{\link{irregFunData}} or \code{\link{multiFunData}}, for that the 
scalar product is to be calculated.}

\item{...}{Additional parameters passed to \code{\link{integrate}}. For
\code{\linkS4class{multiFunData}} objects, one can also pass a
\code{weight} argument. See Details.}
}
\value{
A vector of length \code{nObs(object1)} (or \code{nObs(object2)}, if 
  \code{object1} has only one observation), containing the pairwise scalar 
  product for each observation.
}
\description{
This function calculates the scalar product between two objects of the class 
\code{\linkS4class{funData}}, \code{\linkS4class{irregFunData}} and 
\code{\linkS4class{multiFunData}}. For univariate functions \eqn{f,g} on a
domain \eqn{\mathcal{T}}{T}, the scalar product is defined as 
\deqn{\int_\mathcal{T} f(t) g(t) \mathrm{d}t}{\int_T f(t) g(t) dt} and 
for multivariate functions \eqn{f,g} on domains \eqn{\mathcal{T}_1, \ldots, 
\mathcal{T}_p}{T_1,\ldots,T_p}, it is defined as \deqn{\sum_{j = 1}^p
\int_{\mathcal{T}_j} f^{(j)}(t) g^{(j)}(t) \mathrm{d}t.}{\sum_{j = 1}^p 
\int_T_j f^{(j)}(t) g^{(j)}(t) dt.} As seen in the formula, the objects 
must be defined on the same domain. The scalar product is calculated pairwise
for all observations, thus the objects must also have the same number of 
observations or one object may have only one observation (for which the 
scalar product is calculated with all observations of the other object)). 
Objects of the classes \code{\link{funData}} and \code{\link{irregFunData}} 
can be combined, see \code{\link{integrate}} for details.
}
\details{
For \code{\linkS4class{multiFunData}} one can pass an optional vector
\code{weight} for calculating a weighted scalar product. This vector must
have the same number of elements as the \code{\link{multiFunData}} objects
and have to be non-negative with at least one weight that is different from
0. Defaults to \code{1} for each element. See also \code{\link{norm}}.
}
\examples{
# create two funData objectw with 5 observations on [0,1]
f <- simFunData(N = 5, M = 7, eValType = "linear",
                eFunType = "Fourier", argvals = seq(0,1,0.01))$simData
g <- simFunData(N = 5, M = 4, eValType = "linear",
                eFunType = "Poly", argvals = seq(0,1,0.01))$simData
                
# calculate the scalar product
scalarProduct(f,g)

# the scalar product of an object with itself equals the squared norm
all.equal(scalarProduct(f,f), norm(f, squared = TRUE))

# This works of course also for multiFunData objects...
m <- multiFunData(f,g)
all.equal(scalarProduct(m,m), norm(m, squared = TRUE))

# ...and for irregFunData objects
i <- as.irregFunData(sparsify(f, minObs = 5, maxObs = 10))
all.equal(scalarProduct(i,i), norm(i, squared = TRUE))

# Scalar product between funData and irregFunData objects
scalarProduct(i,f)

# Weighted scalar product for multiFunData objects
scalarProduct(m,m, weight = c(1,2))
}
\seealso{
\code{\link{integrate}}, \code{\link{norm}},
}
