% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/galah_geolocate.R
\name{galah_geolocate}
\alias{galah_geolocate}
\title{Narrow a query to within a specified area}
\usage{
galah_geolocate(..., type = c("polygon", "bbox"))
}
\arguments{
\item{...}{a single \code{sf} object, WKT string or shapefile. Bounding boxes can
be supplied as a \code{tibble}/\code{data.frame} or a \code{bbox}}

\item{type}{\code{string}: one of \code{c("polygon", "bbox")}. Defaults to
\code{"polygon"}. If \code{type = "polygon"}, a multipolygon will be built via
\code{\link[=galah_polygon]{galah_polygon()}}. If \code{type = "bbox"}, a multipolygon will be built via
\code{\link[=galah_bbox]{galah_bbox()}}. The multipolygon is used to narrow a query to the ALA.}
}
\value{
length-1 object of class \code{character} and \code{galah_geolocate},
containing a multipolygon WKT string representing the area provided.
}
\description{
Restrict results to those from a specified area. Areas can be specified as
either polygons or bounding boxes, depending on \code{type}.
}
\details{
By default, \code{type} is set to \code{"polygon"} which narrows queries to within an area
supplied as a \code{POLYGON}. Polygons must be specified as either an \code{sf} object,
a 'well-known text' (WKT) string, or a shapefile. Shapefiles must be simple
to be accepted by the ALA.

Alternatively, set \code{type = "bbox"} to narrow queries to within a
bounding box. Bounding boxes can be extracted from a supplied \code{sf} object or
a shapefile. A bounding box can also be supplied as a \code{bbox} object
(via \code{sf::st_bbox()}) or a \code{tibble}/\code{data.frame}.

If \code{type = "polygon"}, WKT strings longer than 10000 characters and
\code{sf} objects with more than 500 vertices will not be
accepted by the ALA. Some polygons  may need to be simplified. If
\code{type = "bbox"}, sf objects and shapefiles will be converted to a bounding
box to query the ALA.
}
\examples{
\dontrun{
# Search for records within a polygon using a shapefile
location <- sf::st_read("path/to/shapefile.shp")
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(location) |>
  atlas_counts()
  
# Search for records within the bounding box of a shapefile
location <- sf::st_read("path/to/shapefile.shp")
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(location, type = "bbox") |>
  atlas_counts()

# Search for records within a polygon using an `sf` object
location <- "POLYGON((142.3 -29.0,142.7 -29.1,142.7 -29.4,142.3 -29.0))" |>
 sf::st_as_sfc()
galah_call() |>
  galah_identify("reptilia") |>
  galah_polygon(location) |>
  atlas_counts()
   
# Search for records using a Well-known Text string (WKT)
wkt <- "POLYGON((142.3 -29.0,142.7 -29.1,142.7 -29.4,142.3 -29.0))"
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(wkt) |>
  atlas_counts()

# Search for records within the bounding box extracted from an `sf` object
location <- "POLYGON((142.3 -29.0,142.7 -29.1,142.7 -29.4,142.3 -29.0))" |>
  sf::st_as_sfc()
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(location, type = "bbox") |>
  atlas_counts()

# Search for records using a bounding box of coordinates
b_box <- sf::st_bbox(c(xmin = 143, xmax = 148, ymin = -29, ymax = -28), 
                     crs = sf::st_crs("WGS84"))
galah_call() |>
  galah_identify("reptilia") |>
  galah_geolocate(b_box, type = "bbox") |>
  atlas_counts()

# Search for records using a bounding box in a `tibble` or `data.frame`
b_box <- tibble::tibble(xmin = 148, ymin = -29, xmax = 143, ymax = -21)
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(b_box, type = "bbox") |>
  atlas_counts()
}

}
\seealso{
\code{\link[=galah_polygon]{galah_polygon()}} and \code{\link[=galah_bbox]{galah_bbox()}} for specific functions to
narrow queries by a specified area. \code{\link[=search_taxa]{search_taxa()}}, \code{\link[=galah_filter]{galah_filter()}} and
\code{\link[=galah_select]{galah_select()}} for other ways to restrict the information
returned by \code{\link[=atlas_occurrences]{atlas_occurrences()}} and related functions.
}
