% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/07-interpolation.R
\name{spatial_interpolation_comprehensive}
\alias{spatial_interpolation_comprehensive}
\title{Perform spatial interpolation for missing data}
\usage{
spatial_interpolation_comprehensive(
  spatial_data,
  target_variables,
  method = "NN",
  target_grid = NULL,
  region_boundary = NULL,
  power = 2,
  max_distance = Inf,
  min_points = 3,
  max_points = 50,
  cross_validation = FALSE,
  cv_folds = 5,
  handle_outliers = "none",
  outlier_threshold = 3,
  coord_cols = c("lon", "lat"),
  mice_method = "pmm",
  mice_iterations = 10,
  output_format = "sf",
  output_file = NULL,
  verbose = FALSE
)
}
\arguments{
\item{spatial_data}{Spatial data to interpolate. Can be:
\itemize{
\item sf object with point geometries
\item data.frame with coordinate columns
\item File path to spatial data (CSV, SHP, GeoJSON)
}}

\item{target_variables}{Character vector of variables to interpolate}

\item{method}{Interpolation method:
\itemize{
\item \strong{"NN"}: Nearest neighbor (default)
\item \strong{"simple"}: Simple distance weighting
\item \strong{"spline"}: Thin plate spline interpolation
\item \strong{"mice"}: Multivariate imputation (requires mice package)
\item \strong{"auto"}: Automatically select best method based on data
}}

\item{target_grid}{Target grid for interpolation. Can be:
\itemize{
\item SpatRaster template for raster output
\item sf object with target locations
\item NULL for point-to-point interpolation only
}}

\item{region_boundary}{Optional region boundary for clipping results}

\item{power}{Power parameter for simple distance weighting (default: 2)}

\item{max_distance}{Maximum distance for interpolation (map units)}

\item{min_points}{Minimum number of points for interpolation}

\item{max_points}{Maximum number of points to use for each prediction}

\item{cross_validation}{Perform cross-validation for accuracy assessment}

\item{cv_folds}{Number of folds for cross-validation (default: 5)}

\item{handle_outliers}{Method for outlier handling: "none", "remove", "cap"}

\item{outlier_threshold}{Z-score threshold for outlier detection (default: 3)}

\item{coord_cols}{Coordinate column names for data.frame input}

\item{mice_method}{MICE method for multivariate imputation}

\item{mice_iterations}{Number of MICE iterations (default: 10)}

\item{output_format}{Output format: "sf", "raster", "both"}

\item{output_file}{Optional output file path}

\item{verbose}{Print detailed progress messages}
}
\value{
Depending on output_format:
\describe{
\item{"sf"}{sf object with interpolated values}
\item{"raster"}{SpatRaster with interpolated surfaces}
\item{"both"}{List containing both sf and raster results}
}
Additional attributes include:
\itemize{
\item interpolation_info: Method used, parameters, processing time
\item cross_validation: CV results if performed
}
}
\description{
Perform spatial interpolation using reliable methods to fill missing values
in spatial datasets. Supports nearest neighbor, spline interpolation, and
multivariate imputation with comprehensive error handling.
}
\details{
\subsection{Supported Interpolation Methods:}{
\subsection{Distance-Based Methods:}{

\itemize{
\item \strong{NN} (Nearest Neighbor): Assigns nearest known value
- Best for: Categorical data or when preserving exact values
- Fast and creates Voronoi-like patterns
- No assumptions about data distribution

\item \strong{Simple} (Simple distance weighting): Basic distance-based averaging
- Best for: Quick estimates with minimal computation
- Uses inverse distance weighting without external dependencies
}
}

\subsection{Statistical Methods:}{

\itemize{
\item \strong{Spline}: Smooth surface interpolation using thin plate splines
- Best for: Smooth, continuous phenomena
- Creates smooth surfaces without sharp changes
- Good for environmental data with gradual spatial variation
}
}

\subsection{Multivariate Methods:}{

\itemize{
\item \strong{MICE}: Multivariate imputation by chained equations
- Best for: Multiple correlated variables with missing values
- Handles complex missing data patterns
- Preserves relationships between variables
- Requires mice package
}
}

}

\subsection{Input Data Support:}{
\itemize{
\item sf objects with point geometries
\item data.frame with coordinate columns
\item File paths (CSV, shapefile, GeoJSON)
\item Target grids for raster output
}
}

\subsection{Quality Control Features:}{
\itemize{
\item Cross-validation for method comparison
\item Outlier detection and handling
\item Performance metrics calculation
\item Robust error handling
}
}
}
\section{Method Selection Guide}{

\describe{
\item{Dense, regular data}{Simple distance weighting for good balance}
\item{Sparse, irregular data}{Nearest neighbor for stability}
\item{Environmental data}{Spline for smooth surfaces}
\item{Categorical data}{Nearest neighbor}
\item{Multiple correlated variables}{MICE for multivariate patterns}
\item{Unknown data characteristics}{Auto-selection based on data properties}
}
}

\section{Performance Optimization}{

\itemize{
\item For large datasets: Set max_points=50-100 for faster processing
\item For high accuracy: Use cross_validation=TRUE to compare methods
\item For memory efficiency: Process variables individually
\item For smooth results: Use spline method
}
}

\examples{
\dontrun{
# These examples require external data files not included with the package
# Basic nearest neighbor interpolation
soil_interpolated <- spatial_interpolation_comprehensive(
  spatial_data = "soil_samples.csv",
  target_variables = c("nitrogen", "phosphorus", "ph"),
  method = "NN",
  target_grid = study_area_grid,
  region_boundary = "Iowa"
)

# Simple distance weighting
temp_interp <- spatial_interpolation_comprehensive(
  spatial_data = weather_stations,
  target_variables = "temperature",
  method = "simple",
  power = 2,
  cross_validation = TRUE,
  verbose = TRUE
)

# Multivariate imputation for environmental data
env_imputed <- spatial_interpolation_comprehensive(
  spatial_data = env_monitoring,
  target_variables = c("temp", "humidity", "pressure", "wind_speed"),
  method = "mice",
  mice_iterations = 15,
  handle_outliers = "cap"
)

# Auto-method selection with comparison
best_interp <- spatial_interpolation_comprehensive(
  spatial_data = precipitation_data,
  target_variables = "annual_precip",
  method = "auto",
  cross_validation = TRUE,
  cv_folds = 10,
  target_grid = dem_template
)

# Access results and diagnostics
plot(best_interp)  # Plot interpolated surface
best_interp$cross_validation$rmse  # Cross-validation RMSE
best_interp$interpolation_info$method_selected  # Method chosen
}

}
\seealso{
\itemize{
\item \code{\link{universal_spatial_join}} for spatial data integration
\item \code{\link{calculate_spatial_correlation}} for spatial correlation analysis
\item \code{\link{create_spatial_map}} for visualization
}
}
