% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/priors.R
\name{priors}
\alias{priors}
\alias{uniform}
\alias{normal}
\alias{student_t}
\alias{gamma2}
\alias{hs}
\title{Prior distributions}
\source{
Donegan, C., Y. Chun and A. E. Hughes (2020). Bayesian estimation of spatial filters with Moran’s Eigenvectors and hierarchical shrinkage priors. \emph{Spatial Statistics}. \doi{10.1016/j.spasta.2020.100450} (open access: \doi{10.31219/osf.io/fah3z}).

Polson, N.G. and J.G. Scott (2010). Shrink globally, act locally: Sparse Bayesian regularization and prediction. \emph{Bayesian Statistics} 9, 501-538.

Piironen, J and A. Vehtari (2017). Sparsity information and regularization in the horseshoe and other shrinkage priors. In \emph{Electronic Journal of Statistics}, 11(2):5018-5051.
}
\usage{
uniform(lower, upper, variable = NULL)

normal(location = 0, scale, variable = NULL)

student_t(df = 10, location = 0, scale, variable = NULL)

gamma2(alpha, beta, variable = NULL)

hs(global_scale = 1, slab_df = 10, slab_scale, variable = "beta_ev")
}
\arguments{
\item{lower, upper}{lower and upper bounds of the distribution}

\item{variable}{A reserved slot for the variable name; if provided by the user, this may be ignored by \strong{geostan}.}

\item{location}{Location parameter(s), numeric value(s)}

\item{scale}{Scale parameter(s), positive numeric value(s)}

\item{df}{Degrees of freedom, positive numeric value(s)}

\item{alpha}{shape parameter, positive numeric value(s)}

\item{beta}{inverse scale parameter, positive numeric value(s)}

\item{global_scale}{Control the (prior) degree of sparsity in the horseshoe model (0 < global_scale < 1).}

\item{slab_df}{Degrees of freedom for the Student's t model for large coefficients in the horseshoe model (slab_df > 0).}

\item{slab_scale}{Scale parameter for the Student's t model for large coefficients in the horseshoe model (slab_scale > 0).}
}
\value{
An object of class \code{prior} which will be used internally by \strong{geostan} to set parameters of prior distributions.

\subsection{Student's t}{

Return value for \code{student_t} depends on the input; if no arguments are provided (specifically, if the scale parameter is missing), this will return an object of class 'family'; if at least the scale parameter is provided, \code{student_t} will return an object of class \code{prior} containing parameter values for the Student's t distribution.
}
}
\description{
Prior distributions
}
\details{
The prior distribution functions are used to set the values of prior parameters.

Users can control the values of the parameters, but the distribution (model) itself is fixed. The intercept and regression coefficients are given Gaussian prior distributions and scale parameters are assigned Student's t prior distributions. Degrees of freedom parameters are assigned gamma priors, and the spatial autocorrelation parameter in the CAR model, rho, is assigned a uniform prior. The horseshoe (\code{hs}) model is used by \code{\link[geostan]{stan_esf}}.

Note that the \code{variable} argument is used internally by \code{geostan}, and any user provided values will be ignored.
\subsection{Parameterizations}{

For details on how any distribution is parameterized, see the Stan Language Functions Reference document: \url{https://mc-stan.org/users/documentation/}.
}

\subsection{The horseshoe prior}{

The horseshoe prior is used by \code{\link[geostan]{stan_esf}} as a prior for the eigenvector coefficients. The horseshoe model encodes a prior state of knowledge that effectively states, 'I believe a small number of these variables may be important, but I don't know which of them is important.' The horseshoe is a normal distribution with unknown scale (Polson and Scott 2010):

\if{html}{\out{<div class="sourceCode">}}\preformatted{       beta_j ~ Normal(0, tau^2 * lambda_j^2)
}\if{html}{\out{</div>}}

The scale parameter for this prior is the product of two terms: \code{lambda_j^2} is specific to the variable \code{beta_j}, and \code{tau^2} is known as the global shrinkage parameter.

The global shrinkage parameter is assigned a half-Cauchy prior:

\if{html}{\out{<div class="sourceCode">}}\preformatted{       tau ~ Cauchy(0, global_scale * sigma)
}\if{html}{\out{</div>}}

where \code{global_scale} is provided by the user and \code{sigma} is the scale parameter for the outcome variable; for Poisson and binomial models, sigma is fixed at one. Use \code{global_scale} to control the overall sparsity of the model.

The second part of the model is a Student's t prior for \code{lambda_j}. Most \code{lambda_j} will be small, since the model is half-Cauchy:

\if{html}{\out{<div class="sourceCode">}}\preformatted{       lambda_j ~ Cauchy(0, 1)
}\if{html}{\out{</div>}}

This model results in most \code{lambda_j} being small, but due to the long tails of the Cauchy distribution, strong evidence in the data can force any particular \code{lambda_j} to be large. Piironen and Vehtari (2017) adjust the model so that those large \code{lambda_j} are effectively assigned a Student's t model:

\if{html}{\out{<div class="sourceCode">}}\preformatted{       Big_lambda_j ~ Student_t(slab_df, 0, slab_scale)
}\if{html}{\out{</div>}}

This is a schematic representation of the model; see Piironen and Vehtari (2017) or Donegan et al. (2020) for details.
}
}
\examples{
\donttest{
# std. normal priors to k=2 covariates
data(georgia)
prior <- list()
k <- 2
prior$beta <- normal(location = rep(0, times = k),
                     scale = rep(1, times = k))
prior$intercept <- normal(-5, 3)
print(prior)
fit <- stan_glm(deaths.male ~ offset(log(pop.at.risk.male)) + ICE + college,
                re = ~ GEOID,
                data = georgia,
                family = poisson(),
                prior = prior,
                prior_only = TRUE,
                chains = 2, iter = 600) # for speed only
plot(fit)

se <- data.frame(insurance = georgia$insurance.se)
prior <- list()
prior$df <- gamma2(3, 0.2)
prior$location <- normal(50, 50)
prior$scale <- student_t(12, 10, 20)
print(prior)
ME <- prep_me_data(se = se, prior = prior)
fit <- stan_glm(log(rate.male) ~ insurance, 
                data = georgia,
                ME = ME,
                prior_only = TRUE,
               chains = 2, iter = 600) # for speed only
}
}
