% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stan_glm.R
\name{stan_glm}
\alias{stan_glm}
\title{Generalized linear models}
\source{
Donegan, Connor and Chun, Yongwan and Griffith, Daniel A. (2021). Modeling community health with areal data: Bayesian inference with survey standard errors and spatial structure. \emph{Int. J. Env. Res. and Public Health} 18 (13): 6856. DOI: 10.3390/ijerph18136856 Data and code: \url{https://github.com/ConnorDonegan/survey-HBM}.

Donegan, Connor (2021). Building spatial conditional autoregressive (CAR) models in the Stan programming language. \emph{OSF Preprints}. \doi{10.31219/osf.io/3ey65}.
}
\usage{
stan_glm(
  formula,
  slx,
  re,
  data,
  C,
  family = gaussian(),
  prior = NULL,
  ME = NULL,
  centerx = FALSE,
  prior_only = FALSE,
  censor_point,
  chains = 4,
  iter = 2000,
  refresh = 1000,
  keep_all = FALSE,
  slim = FALSE,
  drop = NULL,
  pars = NULL,
  control = NULL,
  ...
)
}
\arguments{
\item{formula}{A model formula, following the R \link[stats]{formula} syntax. Binomial models are specified by setting the left hand side of the equation to a data frame of successes and failures, as in \code{cbind(successes, failures) ~ x}.}

\item{slx}{Formula to specify any spatially-lagged covariates. As in, \code{~ x1 + x2} (the intercept term will be removed internally). When setting priors for \code{beta}, remember to include priors for any SLX terms.}

\item{re}{To include a varying intercept (or "random effects") term, \code{alpha_re}, specify the grouping variable here using formula syntax, as in \code{~ ID}. Then, \code{alpha_re} is a vector of parameters added to the linear predictor of the model, and:

\if{html}{\out{<div class="sourceCode">}}\preformatted{alpha_re ~ N(0, alpha_tau)
alpha_tau ~ Student_t(d.f., location, scale).
}\if{html}{\out{</div>}}}

\item{data}{A \code{data.frame} or an object coercible to a data frame by \code{as.data.frame} containing the model data.}

\item{C}{Optional spatial connectivity matrix which will be used to calculate residual spatial autocorrelation as well as any user specified \code{slx} terms; it will automatically be row-standardized before calculating \code{slx} terms.  See \code{\link[geostan]{shape2mat}}.}

\item{family}{The likelihood function for the outcome variable. Current options are \code{poisson(link = "log")}, \code{binomial(link = "logit")}, \code{student_t()}, and the default \code{gaussian()}.}

\item{prior}{A named list of parameters for prior distributions (see \code{\link[geostan]{priors}}):
\describe{

\item{intercept}{The intercept is assigned a Gaussian prior distribution (see \code{\link[geostan]{normal}}}.

\item{beta}{Regression coefficients are assigned Gaussian prior distributions. Variables must follow their order of appearance in the model \code{formula}. Note that if you also use \code{slx} terms (spatially lagged covariates), and you use custom priors for \code{beta}, then you have to provide priors for the slx terms. Since slx terms are \emph{prepended} to the design matrix, the prior for the slx term will be listed first.
}

\item{sigma}{For \code{family = gaussian()} and \code{family = student_t()} models, the scale parameter, \code{sigma}, is assigned a (half-) Student's t prior distribution. The half-Student's t prior for \code{sigma} is constrained to be positive.}

\item{nu}{\code{nu} is the degrees of freedom parameter in the Student's t likelihood (only used when \code{family = student_t()}). \code{nu} is assigned a gamma prior distribution. The default prior is \code{prior = list(nu = gamma2(alpha = 3, beta = 0.2))}.
}

\item{tau}{The scale parameter for random effects, or varying intercepts, terms. This scale parameter, \code{tau}, is assigned a half-Student's t prior. To set this, use, e.g., \code{prior = list(tau = student_t(df = 20, location = 0, scale = 20))}.}
}}

\item{ME}{To model observational uncertainty (i.e. measurement or sampling error) in any or all of the covariates, provide a list of data as constructed by the \code{\link[geostan]{prep_me_data}} function.}

\item{centerx}{To center predictors on their mean values, use \code{centerx = TRUE}. If the ME argument is used, the modeled covariate (i.e., latent variable), rather than the raw observations, will be centered. When using the ME argument, this is the recommended method for centering the covariates.}

\item{prior_only}{Draw samples from the prior distributions of parameters only.}

\item{censor_point}{Integer value indicating the maximum censored value; this argument is for modeling censored (suppressed) outcome data, typically disease case counts or deaths. For example, the US Centers for Disease Control and Prevention censors (does not report) death counts that are nine or fewer, so if you're using CDC WONDER mortality data you could provide \code{censor_point = 9}.}

\item{chains}{Number of MCMC chains to estimate.}

\item{iter}{Number of samples per chain.}

\item{refresh}{Stan will print the progress of the sampler every \code{refresh} number of samples; set \code{refresh=0} to silence this.}

\item{keep_all}{If \code{keep_all = TRUE} then samples for all parameters in the Stan model will be kept; this is required if you want to do model comparison with Bayes factors and the \code{bridgesampling} package.}

\item{slim}{If \code{slim = TRUE}, then the Stan model will not collect the most memory-intensive parameters (including n-length vectors of fitted values, log-likelihoods, and ME-modeled covariate values). This will disable many convenience functions that are otherwise available for fitted \code{geostan} models, such as the extraction of residuals, fitted values, and spatial trends, WAIC, and spatial diagnostics, and ME diagnostics; many quantities of interest, such as fitted values and spatial trends, can still be calculated manually using given parameter estimates. The "slim" option is designed for data-intensive routines, such as regression with raster data, Monte Carlo studies, and measurement error models. For more control over which parameters are kept or dropped, use the \code{drop} argument instead of \code{slim}.}

\item{drop}{Provide a vector of character strings to specify the names of any parameters that you do not want MCMC samples for. Dropping parameters in this way can improve sampling speed and reduce memory usage. The following parameter vectors can potentially be dropped from GLM models:
\describe{
\item{'fitted'}{The N-length vector of fitted values}
\item{'log_lik'}{The N-length vector of pointwise log-likelihoods, which is used to calculate WAIC.}
\item{'alpha_re'}{Vector of 'random effects'/varying intercepts.}
\item{'x_true'}{N-length vector of 'latent'/modeled covariate values created for measurement error (ME) models.}
}
Using \code{drop = c('fitted', 'log_lik', 'alpha_re', 'x_true')} is equivalent to \code{slim = TRUE}. If \code{slim = TRUE}, then \code{drop} will be ignored.}

\item{pars}{Specify any additional parameters you'd like stored from the Stan model.}

\item{control}{A named list of parameters to control the sampler's behavior. See \link[rstan]{stan} for details.}

\item{...}{Other arguments passed to \link[rstan]{sampling}.}
}
\value{
An object of class class \code{geostan_fit} (a list) containing:
\describe{
\item{summary}{Summaries of the main parameters of interest; a data frame}
\item{diagnostic}{Widely Applicable Information Criteria (WAIC) with a measure of effective number of parameters (\code{eff_pars}) and mean log pointwise predictive density (\code{lpd}), and mean residual spatial autocorrelation as measured by the Moran coefficient.}
\item{stanfit}{an object of class \code{stanfit} returned by \code{rstan::stan}}
\item{data}{a data frame containing the model data}
\item{family}{the user-provided or default \code{family} argument used to fit the model}
\item{formula}{The model formula provided by the user (not including ESF component)}
\item{slx}{The \code{slx} formula}
\item{C}{The spatial weights matrix, if one was provided by the user.}
\item{re}{A list containing \code{re}, the random effects (varying intercepts) formula if provided, and
\code{Data} a data frame with columns \code{id}, the grouping variable, and \code{idx}, the index values assigned to each group.}
\item{priors}{Prior specifications.}
\item{x_center}{If covariates are centered internally (\code{centerx = TRUE}), then \code{x_center} is a numeric vector of the values on which covariates were centered.}
\item{ME}{The \code{ME} data list, if one was provided by the user for measurement error models.}
\item{spatial}{NA, slot is maintained for use in \code{geostan_fit} methods.}
}
}
\description{
Fit a generalized linear model.
}
\details{
Fit a generalized linear model using the R formula interface. Default prior distributions are designed to be weakly informative relative to the data. Much of the functionality intended for spatial models, such as the ability to add spatially lagged covariates and observational error models, are also available in \code{stan_glm}. All of \code{geostan}'s spatial models build on top of the same Stan code used in \code{stan_glm}.
\subsection{Poisson models and disease mapping}{

In spatial statistics, Poisson models are often used to calculate incidence rates (mortality rates, or disease incidence rates) for administrative areas like counties or census tracts. If \eqn{y} are counts of cases, and \eqn{P} are populations at risk, then the crude rates are \eqn{y/P}. The purpose is to model risk \eqn{\eta} for which crude rates are a (noisy) indicator. Our analysis should also respect the fact that the amount of information contained in the observations \eqn{y/P} increases with \eqn{P}. Hierarchical Poisson models are often used to incorporate all of this information.

For the Poisson model, \eqn{y} is specified as the outcome and the log of the population at risk \code{log(P)} needs to be provided as an offset term. For such a case, disease incidence across the collection of areas could be modeled as:

\deqn{y \sim Poisson(e^{log(P) + \eta})}
\deqn{ \eta = \alpha + A}
\deqn{ A \sim Gauss(0, \tau)}
\deqn{\tau \sim Student(20, 0, 2)}
where \eqn{\alpha} is the mean log-risk (incidence rate) and \eqn{A} is a vector of (so-called) random effects, which enable partial pooling of information across observations. Covariates can be added to the model for the log-rates, such that \eqn{\eta = \alpha + X \beta + A}.

Note that the denominator for the rates is specified as a log-offset to provide a consistent, formula-line interface to the model. Using the log-offest (as above) is equivalent to the following:
\deqn{
y \sim Poisson(P * e^{\eta})
}
where \eqn{P} is still the population at risk and it is multiplied by \eqn{e^{\eta}}, the incidence rate (risk).
}

\subsection{Spatially lagged covariates (SLX)}{

The \code{slx} argument is a convenience function for including SLX terms. For example,
\deqn{
 y = W X \gamma + X \beta + \epsilon
}
where \eqn{W} is a row-standardized spatial weights matrix (see \code{\link[geostan]{shape2mat}}), \eqn{WX} is the mean neighboring value of \eqn{X}, and \eqn{\gamma} is a coefficient vector. This specifies a regression with spatially lagged covariates. SLX terms can specified by providing a formula to the \code{slx} argument:

\if{html}{\out{<div class="sourceCode">}}\preformatted{stan_glm(y ~ x1 + x2, slx = ~ x1 + x2, \\...),
}\if{html}{\out{</div>}}

which is a shortcut for

\if{html}{\out{<div class="sourceCode">}}\preformatted{stan_glm(y ~ I(W \\\%*\\\% x1) + I(W \\\%*\\\% x2) + x1 + x2, \\...)
}\if{html}{\out{</div>}}

SLX terms will always be \emph{prepended} to the design matrix, as above, which is important to know when setting prior distributions for regression coefficients.

For measurement error (ME) models, the SLX argument is the only way to include spatially lagged covariates since the SLX term needs to be re-calculated on each iteration of the MCMC algorithm.
}

\subsection{Measurement error (ME) models}{

The ME models are designed for surveys with spatial sampling designs, such as the American Community Survey (ACS) estimates. For a tutorial, see \code{vignette("spatial-me-models", package = "geostan")}.

Given estimates \eqn{x}, their standard errors \eqn{s}, and the target quantity of interest (i.e., the unknown true value) \eqn{z}, the ME models have one of the the following two specifications, depending on the user input. If a spatial CAR model is specified, then:

\deqn{x \sim Gauss(z, s^2)}
\deqn{z \sim Gauss(\mu_z, \Sigma_z)}
\deqn{\Sigma_z = (I - \rho C)^{-1} M}
\deqn{\mu_z \sim Gauss(0, 100)}
\deqn{\tau_z \sim Student(10, 0, 40), \tau > 0}
\deqn{\rho_z \sim uniform(l, u)}

where \eqn{\Sigma} specifies the covariance matrix of a spatial conditional autoregressive (CAR) model with scale parameter \eqn{\tau} (on the diagonal of \eqn{M}), autocorrelation parameter \eqn{\rho}, and \eqn{l}, \eqn{u} are the lower and upper bounds that \eqn{\rho} is permitted to take (which is determined by the extreme eigenvalues of the spatial connectivity matrix \eqn{C}). \eqn{M} contains the inverse of the row sums of \eqn{C} on its diagonal multiplied by \eqn{\tau} (following the "WCAR" specification).

For non-spatial ME models, the following is used instead:
\deqn{x \sim Gauss(z, s^2)}
\deqn{z \sim student_t(\nu_z, \mu_z, \sigma_z)}
\deqn{\nu_z \sim gamma(3, 0.2)}
\deqn{\mu_z \sim Gauss(0, 100)}
\deqn{\sigma_z \sim student(10, 0, 40)}

For strongly skewed variables, such as census tract poverty rates, it can be advantageous to apply a logit transformation to \eqn{z} before applying the CAR or Student-t prior model. When the \code{logit} argument is used, the first two lines of the model specification become:
\deqn{x \sim Gauss(z, s^2)}
\deqn{logit(z) \sim Gauss(\mu_z, \Sigma_z) }
and similarly for the Student t model:
\deqn{x \sim Gauss(z, s^2)}
\deqn{logit(z) \sim student(\nu_z, \mu_z, \sigma_z)}
}

\subsection{Missing data}{

For most geostan models, missing (NA) observations are allowed in the outcome variable. However, there cannot be any missing covariate data. Models that can handle missing data are: any Poisson or binomial model (GLM, SAR, CAR, ESF, ICAR), all GLMs and ESF models. The only models that cannot handle missing outcome data are the CAR and SAR models when the outcome is a continuous variable (auto-normal/Gaussian models).

When observations are missing, they will simply be ignored when calculating the likelihood in the MCMC sampling process (reflecting the absence of information). The estimated model parameters (including any covariates and spatial trend) will then be used to produce estimates or fitted values for the missing observations. The \code{fitted} and \code{posterior_predict} functions will work as normal in this case, and return values for all rows in your data.
}

\subsection{Censored counts}{

Vital statistics systems and disease surveillance programs typically suppress case counts when they are smaller than a specific threshold value. In such cases, the observation of a censored count is not the same as a missing value; instead, you are informed that the value is an integer somewhere between zero and the threshold value. For Poisson models (\verb{family = poisson())}), you can use the \code{censor_point} argument to encode this information into your model.

Internally, \code{geostan} will keep the index values of each censored observation, and the index value of each of the fully observed outcome values. For all observed counts, the likelihood statement will be:
\deqn{
p(y_i | data, model) = poisson(y_i | \mu_i), 
}
as usual, where \eqn{\mu_i} may include whatever spatial terms are present in the model.

For each censored count, the likelihood statement will equal the cumulative Poisson distribution function for values zero through the censor point:
\deqn{
p(y_i | data, model) = \sum_{m=0}^{M} Poisson( m | \mu_i),
}
where \eqn{M} is the censor point and \eqn{\mu_i} again is the fitted value for the \eqn{i^{th}} observation.

For example, the US Centers for Disease Control and Prevention's CDC WONDER database censors all death counts between 0 and 9. To model CDC WONDER mortality data, you could provide \code{censor_point = 9} and then the likelihood statement for censored counts would equal the summation of the Poisson probability mass function over each integer ranging from zero through 9 (inclusive), conditional on the fitted values (i.e., all model parameters). See Donegan (2021) for additional discussion, references, and Stan code.
}
}
\examples{
data(sentencing)

sentencing$log_e <- log(sentencing$expected_sents)
fit.pois <- stan_glm(sents ~ offset(log_e),
                     re = ~ name,
                     family = poisson(),
                     data = sentencing,
                    chains = 2, iter = 800) # for speed only

# MCMC diagnostics plot: Rhat values should all by very near 1
rstan::stan_rhat(fit.pois$stanfit)

# effective sample size for all parameters and generated quantities
# (including residuals, predicted values, etc.)
rstan::stan_ess(fit.pois$stanfit)

# or for a particular parameter
rstan::stan_ess(fit.pois$stanfit, "alpha_re")

# Spatial autocorrelation/residual diagnostics
sp_diag(fit.pois, sentencing)

## Posterior predictive distribution                                       
yrep <- posterior_predict(fit.pois, S = 65)
y <- sentencing$sents
plot(density(yrep[1,]))
for (i in 2:nrow(yrep)) lines(density(yrep[i,]), col = "gray30")
lines(density(sentencing$sents), col = "darkred", lwd = 2)
}
\author{
Connor Donegan, \email{connor.donegan@gmail.com}
}
